<?php

namespace Drupal\turbo_sms_notify\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\webform\WebformInterface;

/**
 * Admin settings form for Turbo SMS Notify.
 */
class SettingsForm extends ConfigFormBase {

  /** {@inheritdoc} */
  public function getFormId(): string {
    return 'turbo_sms_notify_settings_form';
  }

  /** {@inheritdoc} */
  protected function getEditableConfigNames(): array {
    return ['turbo_sms_notify.settings'];
  }

  /** {@inheritdoc} */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('turbo_sms_notify.settings');

    // API & Sender
    $form['api_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API token'),
      '#default_value' => $config->get('api_token'),
      '#required' => TRUE,
    ];
    $form['sender_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sender (alpha name)'),
      '#default_value' => $config->get('sender_name'),
      '#required' => TRUE,
    ];

    // Webform ID
    $form['webform_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Webform machine ID'),
      '#default_value' => $config->get('webform_id'),
      '#required' => TRUE,
      '#description' => $this->t('Machine-name of the Webform. You may use * to match any webform (for testing).'),
    ];

    // Load webform elements options.
    $options = [];
    $webform_id = $form_state->getValue('webform_id') ?: $config->get('webform_id');
    if ($webform_id && $webform_id !== '*') {
      $webform = \Drupal::entityTypeManager()->getStorage('webform')->load($webform_id);
      if ($webform instanceof WebformInterface) {
        $elements = $this->flattenElements($webform->getElementsInitialized());
        foreach ($elements as $key => $element) {
          $options[$key] = $key;
        }
      }
    }

    // Phone field selection.
    $saved_phone_key = (string) ($config->get('phone_field') ?? 'phone');
    $is_saved_known = isset($options[$saved_phone_key]);

    $form['phone_field_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Phone element key'),
      '#description' => $options
        ? $this->t('Choose which element contains phone number.')
        : $this->t('Element list is empty. Save the Webform ID first or type the key manually below.'),
      '#options' => $options,
      '#empty_option' => $this->t('- Enter manually below -'),
      '#default_value' => $is_saved_known ? $saved_phone_key : '',
      '#states' => [
        'visible' => [
          ':input[name="webform_id"]' => ['filled' => TRUE],
        ],
      ],
    ];

    $form['phone_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Phone element key (manual)'),
      '#default_value' => $is_saved_known ? '' : $saved_phone_key,
      '#states' => [
        'visible' => [
          ':input[name="phone_field_select"]' => ['value' => ''],
        ],
      ],
    ];

    // Content & channel.
    $form['message_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message template'),
      '#default_value' => $config->get('message_template'),
      '#description' => $this->t('Use placeholders like [name] and [phone].'),
    ];

    if (\Drupal::hasService('module_handler') && \Drupal::service('module_handler')->moduleExists('token')) {
      $form['message_template']['#description'] = $this->t('Use tokens like [site:name], [current-user:mail], [webform_submission:values:phone]. Unknown tokens will be cleared on send.')
        . ' ' . ($form['message_template']['#description'] ?? '');

      $form['tokens_help'] = [
        '#type' => 'details',
        '#title' => $this->t('Available tokens'),
        '#open' => FALSE,
      ];
      $form['tokens_help']['browser'] = [
        '#type' => 'token_tree_link',
        '#token_types' => ['site', 'webform_submission', 'webform', 'user', 'current-user', 'date'],
      ];
    }

    $form['channel'] = [
      '#type' => 'select',
      '#title' => $this->t('Channel'),
      '#options' => [
        'sms' => $this->t('SMS'),
        'viber_transactional' => $this->t('Viber (transactional)'),
        'viber_promotional' => $this->t('Viber (promotional)'),
      ],
      '#default_value' => $config->get('channel') ?? 'sms',
    ];
    $form['image_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Image URL (Viber optional)'),
      '#default_value' => $config->get('image_url'),
      '#states' => ['visible' => [':input[name="channel"]' => ['!value' => 'sms']]],
    ];
    $form['button_caption'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button caption (Viber)'),
      '#default_value' => $config->get('button_caption'),
      '#states' => ['visible' => [':input[name="channel"]' => ['!value' => 'sms']]],
    ];
    $form['button_action'] = [
      '#type' => 'url',
      '#title' => $this->t('Button action URL (Viber)'),
      '#default_value' => $config->get('button_action'),
      '#states' => ['visible' => [':input[name="channel"]' => ['!value' => 'sms']]],
    ];

    // Webhook secret.
    $form['webhook_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Webhook secret key'),
      '#default_value' => $config->get('webhook_secret'),
      '#description' => $this->t('Must match the secret set in TurboSMS webhook settings (optional).'),
    ];

    // Test block.
    $form['test'] = [
      '#type' => 'details',
      '#title' => $this->t('Send test message'),
      '#open' => TRUE,
    ];
    $form['test']['test_phone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Recipient phone (international format)'),
      '#description' => $this->t('Example: +380671234567'),
    ];
    $form['test']['test_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Test message text'),
      '#default_value' => $this->t('Test message from Drupal TurboSMS.'),
    ];
    $form['test']['send_test'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send test'),
      '#submit' => [[static::class, 'sendTest']],
      '#limit_validation_errors' => [
        ['api_token'],
        ['sender_name'],
        ['channel'],
        ['image_url'],
        ['button_caption'],
        ['button_action'],
        ['test_phone'],
        ['test_message'],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /** {@inheritdoc} */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $phone_field = (string) ($form_state->getValue('phone_field_select') ?: $form_state->getValue('phone_field'));

    $this->config('turbo_sms_notify.settings')
      ->set('api_token', $form_state->getValue('api_token'))
      ->set('sender_name', $form_state->getValue('sender_name'))
      ->set('webform_id', $form_state->getValue('webform_id'))
      ->set('phone_field', $phone_field)
      ->set('message_template', $form_state->getValue('message_template'))
      ->set('channel', $form_state->getValue('channel'))
      ->set('image_url', $form_state->getValue('image_url'))
      ->set('button_caption', $form_state->getValue('button_caption'))
      ->set('button_action', $form_state->getValue('button_action'))
      ->set('webhook_secret', $form_state->getValue('webhook_secret'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Submit callback for the "Send test" button.
   */
  public static function sendTest(array &$form, FormStateInterface $form_state): void {
    /** @var self $form_object */
    $form_object = $form_state->getFormObject();
    $config = $form_object->config('turbo_sms_notify.settings');

    // Prefer unsaved values if present.
    $api_token = (string) ($form_state->getValue('api_token') ?: $config->get('api_token'));
    $sender = (string) ($form_state->getValue('sender_name') ?: $config->get('sender_name'));
    $channel = (string) ($form_state->getValue('channel') ?: $config->get('channel') ?: 'sms');
    $image_url = (string) ($form_state->getValue('image_url') ?: $config->get('image_url'));
    $button_caption = (string) ($form_state->getValue('button_caption') ?: $config->get('button_caption'));
    $button_action = (string) ($form_state->getValue('button_action') ?: $config->get('button_action'));

    $raw_phone = (string) $form_state->getValue('test_phone');
    if ($raw_phone === '') {
      $form_object->messenger()->addError(t('Please provide a phone number for the test.'));
      return;
    }
    $phone = self::normalizePhone($raw_phone);
    if (!$phone) {
      $form_object->messenger()->addError(t('Phone number is invalid. Use international format like +380671234567.'));
      return;
    }

    $text = (string) ($form_state->getValue('test_message') ?: 'Test message from Drupal TurboSMS.');
    if (\Drupal::hasService('token')) {
      $text = \Drupal::token()->replace($text, [], ['clear' => TRUE]);
    }

    $payload = [
      'sender' => $sender,
      'text' => $text,
      'recipients' => [$phone],
    ];
    if ($channel === 'sms') {
      $payload['sms'] = new \stdClass();
    } else {
      $payload['viber'] = [
        'is_transactional' => $channel === 'viber_transactional' ? 1 : 0,
      ];
      if (!empty($image_url)) {
        $payload['viber']['image_url'] = $image_url;
      }
      if (!empty($button_caption)) {
        $payload['viber']['caption'] = $button_caption;
        $payload['viber']['action'] = $button_action;
      }
    }

    try {
      $client = \Drupal::httpClient();
      $response = $client->request('POST', 'https://api.turbosms.ua/message/send.json', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_token,
          'Content-Type' => 'application/json',
        ],
        'json' => $payload,
        'timeout' => 15,
      ]);
      $status = $response->getStatusCode();
      $body = (string) $response->getBody();

      $details = '';
      try {
        $json = $body ? json_decode($body, TRUE, 512, JSON_THROW_ON_ERROR) : NULL;
        if (is_array($json)) {
          if (isset($json['response_code'])) $details = ' (code: ' . $json['response_code'] . ')';
          elseif (isset($json['code'])) $details = ' (code: ' . $json['code'] . ')';
        }
      } catch (\Throwable $e) {}

      if ($status >= 200 and $status < 300) {
        $form_object->messenger()->addStatus(t('Test message sent to @p.@d', ['@p' => $phone, '@d' => $details]));
      } else {
        \Drupal::logger('turbo_sms_notify')->error('TurboSMS test failed: HTTP @s @b', ['@s' => $status, '@b' => $body]);
        $form_object->messenger()->addError(t('Failed to send test message. See logs for details.'));
      }
    } catch (\Throwable $e) {
      \Drupal::logger('turbo_sms_notify')->error('TurboSMS test send error: @m', ['@m' => $e->getMessage()]);
      $form_object->messenger()->addError(t('Error sending test: @m', ['@m' => $e->getMessage()]));
    }
  }

  /**
   * Recursively flatten Webform element tree.
   *
   * @param array $elements
   * @param array $flat
   * @return array
   */
  protected function flattenElements(array $elements, array &$flat = []): array {
    foreach ($elements as $key => $element) {
      if (is_array($element)) {
        $flat[$key] = $element;
        if (isset($element['#webform_composite_elements'])) {
          $this->flattenElements($element['#webform_composite_elements'], $flat);
        }
        foreach ($element as $child_key => $child) {
          if (is_array($child) && strpos($child_key, '#') !== 0) {
            $this->flattenElements([$child_key => $child], $flat);
          }
        }
      }
    }
    return $flat;
  }

  /**
   * Normalize phone to international format with leading '+'.
   *
   * @param string $phone
   * @return string|null
   */
  protected static function normalizePhone(string $phone): ?string {
    if ($phone === '') return NULL;
    $phone = preg_replace('/[^+\d]/', '', $phone) ?? '';
    $digits = preg_replace('/\D/', '', $phone) ?? '';
    if (strlen($digits) < 10) return NULL;
    if ($phone[0] !== '+') $phone = '+' . $digits;
    return $phone;
  }

}
