<?php

namespace Drupal\turbo_sms_notify\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;

/**
 * Sends messages via TurboSMS (SMS or Viber) using module config.
 */
class Sender {

  /** @var \GuzzleHttp\ClientInterface */
  protected ClientInterface $http;

  /** @var \Drupal\Core\Config\ImmutableConfig */
  protected \Drupal\Core\Config\ImmutableConfig $cfg;

  /** @var \Psr\Log\LoggerInterface */
  protected \Psr\Log\LoggerInterface $log;

  /**
   * Constructor with DI.
   *
   * @param \GuzzleHttp\ClientInterface $http
   *   HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory.
   */
  public function __construct(
    ClientInterface $http,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->http = $http;
    $this->cfg  = $config_factory->get('turbo_sms_notify.settings');
    $this->log  = $logger_factory->get('turbo_sms_notify');
  }

  /**
   * Send a single message to a phone number.
   *
   * @param string $phone
   *   E.164-like phone with leading '+'.
   * @param string $text
   *   Message text.
   * @param array $overrides
   *   Optional overrides: ['channel' => 'sms'|'viber_transactional'|'viber_promotional',
   *   'image_url' => '...', 'button_caption' => '...', 'button_action' => '...'].
   *
   * @return bool
   *   TRUE on HTTP 2xx, FALSE otherwise.
   */
  public function send(string $phone, string $text, array $overrides = []): bool {
    $token   = (string) ($this->cfg->get('api_token') ?? '');
    $sender  = (string) ($this->cfg->get('sender_name') ?? '');
    $channel = (string) ($overrides['channel'] ?? $this->cfg->get('channel') ?? 'sms');

    if ($token === '' || $sender === '') {
      $this->log->warning('Sender: missing API token or sender.');
      return FALSE;
    }

    $payload = [
      'sender'     => $sender,
      'text'       => $text,
      'recipients' => [$phone],
    ];

    if ($channel === 'sms') {
      $payload['sms'] = new \stdClass();
    }
    else {
      $payload['viber'] = [
        'is_transactional' => $channel === 'viber_transactional' ? 1 : 0,
      ];
      $image_url     = (string) ($overrides['image_url']     ?? $this->cfg->get('image_url') ?? '');
      $button_cap    = (string) ($overrides['button_caption']?? $this->cfg->get('button_caption') ?? '');
      $button_action = (string) ($overrides['button_action'] ?? $this->cfg->get('button_action') ?? '');

      if ($image_url !== '') {
        $payload['viber']['image_url'] = $image_url;
      }
      if ($button_cap !== '') {
        $payload['viber']['caption'] = $button_cap;
        $payload['viber']['action']  = $button_action;
      }
    }

    try {
      $res = $this->http->request('POST', 'https://api.turbosms.ua/message/send.json', [
        'headers' => [
          'Authorization' => 'Bearer ' . $token,
          'Content-Type'  => 'application/json',
        ],
        'json'    => $payload,
        'timeout' => 15,
      ]);

      $ok = ($res->getStatusCode() >= 200 && $res->getStatusCode() < 300);
      $this->log->info('Sender: ' . ($ok ? 'queued' : ('HTTP '.$res->getStatusCode())) . ' to @p', ['@p' => $phone]);
      return $ok;
    }
    catch (\Throwable $e) {
      $this->log->error('Sender: exception to @p: @m', ['@p' => $phone, '@m' => $e->getMessage()]);
      return FALSE;
    }
  }

}
