<?php

namespace Drupal\turbo_sms_notify\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Receives TurboSMS webhook callbacks (message status updates, etc.).
 */
class WebhookController implements ContainerInjectionInterface {

  protected \Drupal\Core\Config\ImmutableConfig $config;
  protected LoggerInterface $logger;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->config = $config_factory->get('turbo_sms_notify.settings');
    $this->logger = $logger_factory->get('turbo_sms_notify');
  }

  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('logger.factory')
    );
  }

  /**
   * Handle incoming webhook from TurboSMS.
   */
  public function receive(Request $request): JsonResponse {
    if ($request->getMethod() !== 'POST') {
      return new JsonResponse(['error' => 'Method Not Allowed'], 405);
    }

    $expected = (string) ($this->config->get('webhook_secret') ?? '');
    if ($expected !== '') {
      $provided = (string) ($request->headers->get('X-TurboSMS-Signature') ?? '');
      if (!hash_equals($expected, $provided)) {
        $this->logger->warning('Webhook rejected due to invalid signature.');
        return new JsonResponse(['error' => 'Forbidden'], 403);
      }
    }

    $raw = $request->getContent() ?: '';
    try {
      $data = $raw !== '' ? json_decode($raw, TRUE, 512, JSON_THROW_ON_ERROR) : [];
    } catch (\Throwable $e) {
      $this->logger->error('Webhook JSON parse error: @m', ['@m' => $e->getMessage()]);
      return new JsonResponse(['error' => 'Bad Request'], 400);
    }

    $this->logger->info('TurboSMS webhook received: @payload', [
      '@payload' => substr($raw, 0, 2000),
    ]);

    return new JsonResponse(['status' => 'ok']);
  }

}
