<?php

namespace Drupal\tv\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\node\Controller\NodeViewController;
use Drupal\tv\Service\TvChannelServiceInterface;
use Drupal\tv\Channel\ChannelContextCollector;
use Psr\Container\ContainerInterface;

/**
 * Provides a 'TV' block.
 *
 * @Block(
 *   id = "tv",
 *   admin_label = @Translation("TV Block"),
 *   category = @Translation("Media")
 * )
 */
class TVBlock extends BlockBase implements ContainerFactoryPluginInterface {

    private TvChannelServiceInterface $tvChannelService;
    private RouteMatchInterface $routeMatch;
    private ChannelContextCollector $channelContextCollector;


    public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $routeMatch, TvChannelServiceInterface $tvChannelService, ChannelContextCollector $channelContextCollector)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->tvChannelService = $tvChannelService;
        $this->routeMatch = $routeMatch;
        $this->channelContextCollector = $channelContextCollector;
    }

    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('current_route_match'),
            $container->get('tv.channel'),
            $container->get('tv.channel_context_collector')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function build() {
        $build = [];
        // Try to get channels from context providers first, fallback to routeMatch.
        $channels = $this->channelContextCollector->getChannelNodes();
        if (empty($channels)) {
            $channels = $this->getChannels();
        }
        if (empty($channels)) {
            return $build;
        }

        // Don't show the TV if there are no channels with items.
        $itemCount = 0;
        foreach ($channels as $channel) {
            $itemCount += count($this->tvChannelService->getItemIds($channel));
        }
        if ($itemCount == 0) {
            return $build;
        }

        // Build attributes.
        $attributes['class'] = 'tv';
        $attributes['data-base-url'] = \Drupal::request()->getSchemeAndHttpHost();
        $attributes['data-channel'] = $channels[0]->id();
        // @todo support multiple channels here

        // Build the block.
        $build[] = [
            '#type' => 'container',
            '#attributes' => $attributes,
            '#attached' => [
                'library' => [
                    $this->isDev() ? 'tv/tv_dev' : 'tv/tv_prod',
                ],
            ],
        ];
        return $build;
    }

    private function isDev(): bool {
        // @todo improve dev instance detection
        return getenv('DDEV_PROJECT') !== false;
    }

    private function getChannels(): array {
        $channels = [];
        $route = $this->routeMatch->getRouteObject();
        if (!$route) {
            throw new \Exception('No route available');
        }
        $controller = $route->getDefault('_controller');
        $node = $this->routeMatch->getParameter('node');
        if ($controller != '\\' . NodeViewController::class . '::view' || !$node) {
            throw new \Exception('Not a node view route');
        }
        $bundle = $node->bundle();
        if ($bundle != 'tv_channel') {
            return $channels;
        }
        $channels[] = $node;
        return $channels;
    }

    public function getCacheTags() {
        if ($node = $this->routeMatch->getParameter('node')) {
            return Cache::mergeTags(parent::getCacheTags(), ['node:' . $node->id()]);
        } else {
            return parent::getCacheTags();
        }
    }

    public function getCacheContexts() {
        return Cache::mergeContexts(parent::getCacheContexts(), ['route']);
    }

    public function getCacheMaxAge() {
        return 0;
    }

}
