import { useEffect, useCallback } from 'react';
import './Channel.css';
import './ChannelItems.css';
import tvStateMachine from '../tvStateMachine';
import AccordionMenu from './AccordionMenu';
import { formatDuration } from '../helpers';

// Custom hook to fetch channel when currentChannelId changes
export function useChannelFetch(currentChannelId, getChannel) {
    useEffect(() => {
        if (currentChannelId) {
            getChannel();
        }
    }, [currentChannelId, getChannel]);
}

// Handler to reset progress for an item
export const handleResetProgress = (channel, updateProgress, baseUrl, currentItem, tvRef) => (item, index) => {
    if (typeof item === 'object' && typeof index === 'number' && item.id) {
        // Always reset progress for the reset button
        fetch(baseUrl + '/tv/progress', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({media_id: item.id, progress: 0})
        }).catch(e => console.error('Progress reset failed:', e));
        updateProgress(item.id, 0);
        // If this is the currently playing item, seek to beginning
        if (index === currentItem && tvRef.current && tvRef.current.players && tvRef.current.players.youtube && typeof tvRef.current.players.youtube.seekTo === 'function') {
            tvRef.current.players.youtube.seekTo(0);
        }
    }
};

// Custom hook to sync playlist length, currentItem, and handle user selection
export function useChannelStateSync(channel, setCurrentItem, setCurrentFocusIndex, openIndex) {
  // Sync playlist length to state machine
  useEffect(() => {
    try {
      const length = channel.playlist ? channel.playlist.length : 0;
      if (length > 0 && tvStateMachine.getState().playlistLength !== length) {
        tvStateMachine.transition('SET_PLAYLIST_LENGTH', length);
      }
    } catch (e) {
      console.error('Error syncing playlist length:', e);
    }
  }, [channel.playlist]);

  // Sync currentItem from state machine
  useEffect(() => {
    try {
      const unsub = tvStateMachine.subscribe((event, prev, next) => {
        try {
          if (prev.currentItem !== next.currentItem) {
            setCurrentItem(next.currentItem);
          }
        } catch (e) {
          console.error('Error in channel state sync subscription:', e);
        }
      });
      // Set initial value
      setCurrentItem(tvStateMachine.getState().currentItem);
      return unsub;
    } catch (e) {
      console.error('Error setting up channel state sync:', e);
      return () => {};
    }
  }, [setCurrentItem]);

  return {};
}// Custom hook to manage channel API
export function useChannelApi(baseUrl, currentChannelId, setChannel, setChannels) {
    const getChannel = useCallback(() => {
        if (!currentChannelId) {
            return;
        }
        fetch(baseUrl + '/node/' + currentChannelId + '/channel')
            .then(response => {
                return response.json();
            })
            .then(data => {
                setChannel(data);
            })
            .catch(error => {
                console.error('[useChannelApi] Error fetching channel:', error);
            });
    }, [baseUrl, currentChannelId, setChannel]);

    const getChannels = useCallback(() => {
        fetch(baseUrl + '/tv/channels')
            .then(response => {
                return response.json();
            })
            .then(data => {
                // Ensure data is an array before setting
                if (Array.isArray(data.channels)) {
                    setChannels(data.channels);
                } else {
                    console.error('[useChannelApi] Expected channels to be an array, got:', data);
                    setChannels([]);
                }
            })
            .catch(error => {
                console.error('[useChannelApi] Error fetching channels:', error);
                setChannels([]);
            });
    }, [baseUrl, setChannels]);

    return { getChannel, getChannels };
}

// Handler to set current item and reset progress if needed
export const setCurrentItemAndResetIfNeeded = (channel, updateProgress, baseUrl, setCurrentItem) => (index, itemIndex) => {
    const item = channel.playlist[itemIndex];
    if (item && typeof item.progress === 'number' && item.progress >= 95 && item.id) {
        // Reset progress for fully played videos
        fetch(baseUrl + '/tv/progress', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({media_id: item.id, progress: 0})
        }).catch(e => console.error('Progress reset failed:', e));
        updateProgress(item.id, 0);
    }
    setCurrentItem(index);
};

function ChannelItems({ channel, currentItem, onItemSelect, handleResetProgress, focused, headingRef, open, setOpen, scrollableHeight, getFocusState }) {
    // Try different possible property names for playlist
    const playlist = channel?.playlist || channel?.videos || channel?.items || [];

    return (
        <AccordionMenu heading="Channel" className="channel-menu" focused={focused} headingRef={headingRef} open={open} setOpen={setOpen} scrollableHeight={scrollableHeight}>
            <ul>
                {(() => {
                    const noItems = !channel || !Array.isArray(playlist) || playlist.length === 0;
                    return noItems;
                })() && (
                    <li className="channel-items-empty">
                        No items in this channel
                    </li>
                )}
                {playlist.map((item, index) => {
                    return (
                        <li key={item?.id || index} className={`${index === currentItem ? 'current' : ''} ${getFocusState ? (getFocusState('channel', index, false) ? 'focused' : '') : ''}`}>
                            <button
                                onClick={e => {
                                    e.preventDefault();
                                    if (typeof onItemSelect === 'function') {
                                        onItemSelect(index);
                                    }
                                }}
                                className={`channel-item-button${index === currentItem ? ' current' : ''}`}
                            >
                                {item?.posterUrl && (
                                    <img src={item.posterUrl} alt={item?.name || 'Video'} />
                                )}
                                {item?.name || 'Untitled'}
                            </button>
                        <p className="channel-item-duration">Duration: {formatDuration(typeof item?.duration === 'number' ? item.duration : (typeof item?.duration === 'string' ? parseInt(item.duration, 10) : 0))}</p>
                        <div className="progress-row">
                            <button
                                className="progress-refresh-btn"
                                title="Reset Progress"
                                aria-label="Reset Progress"
                                onClick={e => {
                                    e.preventDefault();
                                    if (typeof handleResetProgress === 'function') {
                                        handleResetProgress(item, index);
                                    }
                                }}
                            >
                                &#8635;
                            </button>
                            <div className="progress-container" style={{flex:1}}>
                                <div className="progress-bar" style={{width: `${typeof item?.progress === 'number' ? item.progress : 0}%`}}></div>
                            </div>
                        </div>
                    </li>
                );
                })}
            </ul>
        </AccordionMenu>
    );
}

export default ChannelItems;
