import React from 'react';

import './Accordion.css';

/**
 * Accordion: Manages open/close state for multiple AccordionMenu children.
 * Usage:
 * <Accordion>
 *   <AccordionMenu heading="...">...</AccordionMenu>
 *   <AccordionMenu heading="...">...</AccordionMenu>
 * </Accordion>
 */

import { useRef, useLayoutEffect, useState, useMemo } from 'react';

function Accordion({ children, openIndex, setOpenIndex, defaultOpenIndex = 0 }) {
  const [internalOpenIndex, setInternalOpenIndex] = React.useState(defaultOpenIndex);
  const currentOpenIndex = openIndex !== undefined ? openIndex : internalOpenIndex;
  const currentSetOpenIndex = setOpenIndex || setInternalOpenIndex;
  const containerRef = useRef(null);
  const [containerHeight, setContainerHeight] = useState(undefined);
  const [headingHeights, setHeadingHeights] = useState([]);

  // Gather heading refs from children - persist across renders
  const headingRefs = useMemo(() =>
    React.Children.map(children, () => React.createRef()) || [],
    [children]
  );

  useLayoutEffect(() => {
    function updateHeights() {
      if (containerRef.current) {
        setContainerHeight(containerRef.current.offsetHeight);
        setHeadingHeights(
          headingRefs.map(ref => ref.current ? ref.current.offsetHeight : 0)
        );
      }
    }
    updateHeights();
    window.addEventListener('resize', updateHeights);
    return () => window.removeEventListener('resize', updateHeights);
  }, [children, currentOpenIndex, headingRefs]);

  // Clone children and inject open/setOpen/headingRef/scrollableHeight props
  const menus = React.Children.map(children, (child, idx) => {
    if (!React.isValidElement(child)) return child;
    const isOpen = currentOpenIndex === idx;
    let scrollableHeight = undefined;
    if (isOpen && containerHeight && headingHeights[idx] !== undefined) {
      scrollableHeight = Math.max(containerHeight - headingHeights[idx], 0);
    }
    return React.cloneElement(child, {
      open: isOpen,
      setOpen: () => {
        currentSetOpenIndex(idx);
      },
      headingRef: headingRefs[idx],
      scrollableHeight,
    });
  });

  return <div className="accordion" ref={containerRef}>{menus}</div>;
}

export default Accordion;
