import { useEffect, useCallback } from 'react';
import tvStateMachine from '../tvStateMachine';
import { keysEvents } from './Keys';
import './Controls.css';

// Controls component expects tvRef and playerState as props
// Handles all key and button events for player controls and state machine
function Controls({ tvRef, containerRef, playerState, fullscreen, size = 'normal', controlsFocusIdx = -1 }) {
    // Play/Pause handler
    const handlePlayPause = useCallback((e) => {
        if (e) e.preventDefault();
        const player = tvRef.current && tvRef.current.defaultPlayer;
        if (player && typeof player.getPlayerState === 'function') {
            const state = player.getPlayerState();
            if (state === 1) {
                player.pauseVideo();
                tvStateMachine.transition('PAUSE');
            } else {
                player.playVideo();
                tvStateMachine.transition('PLAY');
            }
        }
    }, [tvRef]);

    // Mute/Unmute handler
    const handleMuteUnmute = useCallback((e) => {
        if (e) e.preventDefault();
        if (
            tvRef.current &&
            tvRef.current.defaultPlayer &&
            typeof tvRef.current.defaultPlayer.isMuted === 'function'
        ) {
            const muted = tvRef.current.defaultPlayer.isMuted();
            if (muted) {
                tvRef.current.defaultPlayer.unMute();
                tvStateMachine.transition('UNMUTE');
            } else {
                tvRef.current.defaultPlayer.mute();
                tvStateMachine.transition('MUTE');
            }
        } else {
            console.warn('[Controls] tvRef or defaultPlayer not available for mute/unmute');
        }
    }, [tvRef]);

    // Fullscreen handler
    const handleFullscreen = useCallback((e) => {
        if (e) e.preventDefault();
        if (fullscreen) {
            // Exit fullscreen
            if (document.exitFullscreen) {
                document.exitFullscreen().then(() => {
                    // Update state machine
                    tvStateMachine.transition('FULLSCREEN_EXIT');
                }).catch(e => {
                    console.error('Fullscreen exit failed:', e);
                });
            } else if (document.webkitExitFullscreen) {
                document.webkitExitFullscreen();
                tvStateMachine.transition('FULLSCREEN_EXIT');
            } else if (document.mozCancelFullScreen) {
                document.mozCancelFullScreen();
                tvStateMachine.transition('FULLSCREEN_EXIT');
            } else if (document.msExitFullscreen) {
                document.msExitFullscreen();
                tvStateMachine.transition('FULLSCREEN_EXIT');
            } else {
                // Fallback: just update state machine
                tvStateMachine.transition('FULLSCREEN_EXIT');
            }
        } else {
            // Enter fullscreen
            if (containerRef && containerRef.current && containerRef.current.requestFullscreen) {
                try {
                    const result = containerRef.current.requestFullscreen();
                    if (result && typeof result.catch === 'function') {
                        result.catch(e => {
                            console.error('Fullscreen request failed:', e, containerRef.current);
                        });
                    }
                } catch (e) {
                    console.error('Fullscreen request threw:', e, containerRef.current);
                }
            } else {
                alert('Could not find TV app container to fullscreen.');
                console.error('No containerRef or requestFullscreen not supported', containerRef);
            }
            // Notify state machine
            tvStateMachine.transition('FULLSCREEN_ENTER');
        }
    }, [fullscreen, containerRef]);

    useEffect(() => {
        // Play/Pause
        const unsubPlayPause = keysEvents.on('play_pause', () => {
            const player = tvRef.current && tvRef.current.defaultPlayer;
            if (player && typeof player.getPlayerState === 'function') {
                const state = player.getPlayerState();
                if (state === 1) {
                    player.pauseVideo();
                    tvStateMachine.transition('PAUSE');
                } else {
                    player.playVideo();
                    tvStateMachine.transition('PLAY');
                }
            }
        });
        // Mute/Unmute
        const unsubMuteToggle = keysEvents.on('mute_toggle', () => {
            const player = tvRef.current && tvRef.current.defaultPlayer;
            if (player && typeof player.isMuted === 'function') {
                if (player.isMuted()) {
                    player.unMute();
                    tvStateMachine.transition('UNMUTE');
                } else {
                    player.mute();
                    tvStateMachine.transition('MUTE');
                }
            }
        });
        // Scrub forward +30s
        const unsubScrubForward = keysEvents.on('scrub_coarse_forward', () => {
            const player = tvRef.current && tvRef.current.defaultPlayer;
            if (player && typeof player.getCurrentTime === 'function' && typeof player.seekTo === 'function') {
                const currentTime = player.getCurrentTime();
                player.seekTo(currentTime + 30);
            }
        });
        // Scrub back -30s
        const unsubScrubBack = keysEvents.on('scrub_coarse_back', () => {
            const player = tvRef.current && tvRef.current.defaultPlayer;
            if (player && typeof player.getCurrentTime === 'function' && typeof player.seekTo === 'function') {
                const currentTime = player.getCurrentTime();
                player.seekTo(Math.max(0, currentTime - 30));
            }
        });
        // Control select for navigation
        const unsubControlSelect = keysEvents.on('control_select', (index) => {
            if (index === 0) {
                handlePlayPause();
            } else if (index === 1) {
                handleMuteUnmute();
            } else if (index === 2) {
                handleFullscreen();
            }
        });
        return () => {
            unsubPlayPause();
            unsubMuteToggle();
            unsubScrubForward();
            unsubScrubBack();
            unsubControlSelect();
        };
    }, [tvRef, handlePlayPause, handleMuteUnmute, handleFullscreen]);

    // Render controls bar
    const iconSize = size === 'large' ? 32 : 28;
    const buttons = [
        <button key="playpause" className={`tv-btn-square${0 === controlsFocusIdx ? ' focused' : ''}`} onClick={handlePlayPause} title="Play/Pause">
            {playerState === 'playing' ? (
                // Pause glyph
                <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><rect x="6" y="4" width="4" height="16"/><rect x="14" y="4" width="4" height="16"/></svg>
            ) : (
                // Play glyph
                <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><polygon points="5 3 19 12 5 21 5 3"/></svg>
            )}
        </button>,
        <button key="mute" className={`tv-btn-square${1 === controlsFocusIdx ? ' focused' : ''}`} onClick={handleMuteUnmute} title="Mute/Unmute">
            {(() => {
                try {
                    return tvRef.current && tvRef.current.defaultPlayer && typeof tvRef.current.defaultPlayer.isMuted === 'function' && tvRef.current.defaultPlayer.isMuted() ? (
                        // Muted glyph
                        <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><polygon points="1 1 23 23"/><path d="M9 9v6h4l5 5V4l-5 5H9z"/></svg>
                    ) : (
                        // Unmuted glyph
                        <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><polygon points="9 9 9 15 13 15 18 20 18 4 13 9 9 9"/></svg>
                    );
                } catch (e) {
                    console.error('Error checking mute state:', e);
                    return <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><polygon points="9 9 9 15 13 15 18 20 18 4 13 9 9 9"/></svg>;
                }
            })()}
        </button>,
        <button key="fullscreen" className={`tv-btn-square${2 === controlsFocusIdx ? ' focused' : ''}`} onClick={handleFullscreen} title={fullscreen ? 'Exit Fullscreen' : 'Fullscreen'}>
            {fullscreen ? (
                // Collapse/minify glyph (exit fullscreen)
                <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
                    <polyline points="4 14 10 14 10 20" />
                    <polyline points="20 10 14 10 14 4" />
                    <line x1="14" y1="10" x2="21" y2="3" />
                    <line x1="10" y1="14" x2="3" y2="21" />
                </svg>
            ) : (
                // Expand/fullscreen glyph
                <svg width={iconSize} height={iconSize} viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"><path d="M15 3h6v6"/><path d="M9 21H3v-6"/><path d="M21 3l-7.5 7.5"/><path d="M3 21l7.5-7.5"/></svg>
            )}
        </button>,
    ];
    return <div className="tv-controls-buttons" style={{ display: 'flex', flexDirection: 'row', alignItems: 'center', justifyContent: 'center', gap: '0.5em' }}>{buttons}</div>;
}

export default Controls;
