import { useEffect } from 'react';

/**
 * Mouse and Keyboard activity tracker for idle/active UI state.
 *
 * Props:
 * - tvRef: ref to the main TV container
 * - isHovered: boolean, is mouse over the TV
 * - setIsHovered: function, set hover state
 * - isIdle: boolean, is mouse idle
 * - setIsIdle: function, set idle state
 */
function Mouse({ tvRef, isHovered, setIsHovered, isIdle, setIsIdle }) {
  useEffect(() => {
    const tv = tvRef.current;
    if (!tv) return;
    let idleTimeout = null;

    const resetIdleTimer = () => {
      if (!isHovered) setIsHovered(true);
      if (isIdle) {
        setIsIdle(false);
      }
      if (idleTimeout) clearTimeout(idleTimeout);
      idleTimeout = setTimeout(() => {
        if (!isIdle) {
          setIsIdle(true);
        }
      }, 5000);
    };

    const handleMouseMove = (e) => {
      resetIdleTimer();
    };

    const handleKeyDown = (e) => {
      // Reset idle timer on any key press
      resetIdleTimer();
    };

    tv.addEventListener('mousemove', handleMouseMove);
    tv.addEventListener('keydown', handleKeyDown);

    // Start idle timer if not idle
    idleTimeout = setTimeout(() => {
      if (!isIdle) {
        setIsIdle(true);
      }
    }, 5000);

    return () => {
      tv.removeEventListener('mousemove', handleMouseMove);
      tv.removeEventListener('keydown', handleKeyDown);
      if (idleTimeout) clearTimeout(idleTimeout);
    };
  }, [tvRef, isHovered, setIsHovered, isIdle, setIsIdle]);

  return null;
}

export default Mouse;
