import {getInSeconds, parseDuration} from "../../../../helpers";
import tvStateMachine from '../../../../tvStateMachine';
import React, { useEffect, useRef, useCallback } from 'react';

function YouTube(params) {
    // Import playerState from state machine
    const playerState = tvStateMachine.getState().playerState;
    const item = params.channel.playlist[params.item] || { id: '', url: '', duration: 0 };

    const updateProgress = params.updateProgress;
    const onVideoEnd = params.onVideoEnd;
    const currentMediaIdRef = useRef(item.id);
    const channelRef = useRef(params.channel);
    const videoRef = useRef(item);
    const durationSecondsRef = useRef(typeof videoRef.current.duration === 'number' ? videoRef.current.duration : parseDuration(videoRef.current.duration));
    // Always call useRef, then use prop if provided
    const localTvRef = useRef(null);
    const tvRef = params.tvRef || localTvRef;
    const loadTimestampsRef = useRef([]);
    const currentVideoIdRef = useRef(getVideoId(item.url));

    useEffect(() => {
        channelRef.current = params.channel;
        videoRef.current = params.channel.playlist[params.item];
        durationSecondsRef.current = typeof videoRef.current.duration === 'number' ? videoRef.current.duration : parseDuration(videoRef.current.duration);
        currentMediaIdRef.current = videoRef.current.id;
    }, [params.channel, params.item]);

    // Sync quality changes
    useEffect(() => {
        const unsubscribe = tvStateMachine.subscribe((event, prev, next) => {
            if (prev.quality !== next.quality) {
                const tv = tvRef.current;
                if (tv && tv.players && tv.players.youtube) {
                    const player = tv.players.youtube;
                    if (typeof player.getCurrentTime === 'function' && typeof player.loadVideoById === 'function' && typeof player.getVideoData === 'function') {
                        const currentTime = player.getCurrentTime();
                        const videoData = player.getVideoData();
                        if (videoData && videoData.video_id) {
                            const ytQuality = next.quality === 'low' ? 'small' : next.quality === 'med' ? 'large' : 'auto';
                            player.loadVideoById({videoId: videoData.video_id, startSeconds: currentTime, suggestedQuality: ytQuality});
                        }
                    }
                }
            }
        });
        return unsubscribe;
    }, [tvRef]);

    // Prevent rapid reloads: only load if videoId or item.id is different
    const lastLoadedIdRef = useRef(null);
    useEffect(() => {
        const tv = tvRef.current;
        const playlist = channelRef.current.playlist || [];
        const vid = playlist[params.item];
        if (!vid) return;
        const videoId = getVideoId(vid.url);
        // Only run if the videoId or item.id is different from last loaded
        if (lastLoadedIdRef.current === vid.id) return;
        lastLoadedIdRef.current = vid.id;
        if (tv && tv.players && tv.players.youtube && tv.players.youtube.load) {
            const now = Date.now();
            // Rate limiting: allow channel surfing but prevent abuse
            // Keep only timestamps from the last 10 seconds
            loadTimestampsRef.current = loadTimestampsRef.current.filter(t => now - t < 10000);
            // Error if more than 10 loads in the last 10 seconds
            if (loadTimestampsRef.current.length >= 10) {
                throw new Error('Player ID changed/reloaded too frequently.');
            }
            loadTimestampsRef.current.push(now);
            
            const duration = typeof vid.duration === 'number' ? vid.duration : parseDuration(vid.duration);
            const progress = vid.progress || 0;
            const remaining = duration - ((progress / 100) * duration);
            // If the video has less than 30s left, do not load or play
            if (duration > 0 && remaining < 30) {
                tvStateMachine.transition('END');
                return;
            }
            let startSeconds = progress > 10 ? Math.round((progress / 100) * duration) : 0;
            if (startSeconds > duration - 10) startSeconds = 0;
            currentVideoIdRef.current = videoId;
            tv.players.youtube.loadVideoById(videoId, startSeconds);
            // Only autoplay if not at end
            setTimeout(() => {
                const player = tv.players.youtube;
                if (player && typeof player.getDuration === 'function' && typeof player.getCurrentTime === 'function') {
                    const duration = player.getDuration();
                    const currentTime = player.getCurrentTime();
                    // If video is near end, do not play
                    if (duration > 0 && currentTime > duration - 2) {
                        return;
                    }
                }
                // Only play if not already playing and not at end
                if (
                    playerState === 'playing' &&
                    typeof player.playVideo === 'function' &&
                    typeof player.getPlayerState === 'function' &&
                    player.getPlayerState() !== 1 && // 1 = PLAYING
                    (typeof window.YT === 'undefined' || player.getPlayerState() !== window.YT.PlayerState.ENDED) // Don't play if ended
                ) {
                    player.playVideo();
                }
            }, 300);
        }
    }, [params.item, params.channel, tvRef, playerState]);

    function getVideoId(url) {
        if (!url || typeof url !== 'string') return '';
        const videoId = url.split('v=')[1];
        if (!videoId) return '';
        const ampersandPosition = videoId.indexOf('&');
        if (ampersandPosition !== -1) {
            return videoId.substring(0, ampersandPosition);
        }
        return videoId;
    }

    // Load the YouTube API if not already loaded.
    if (!document.getElementById('iframe-demo')) {
        let tag = document.createElement('script');
        tag.id = 'iframe-demo';
        tag.src = 'https://www.youtube.com/iframe_api';
        let firstScriptTag = document.getElementsByTagName('script')[0];
        firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
    }

    // Configure the YouTube API & TV Player embeds.
    const handleStateChange = useCallback((event, tv) => {
        // Only act if the current videoId matches the intended one
        const playerVideoId = tv && tv.players && tv.players.youtube && typeof tv.players.youtube.getVideoData === 'function'
            ? tv.players.youtube.getVideoData().video_id
            : null;
        if (playerVideoId && playerVideoId !== currentVideoIdRef.current) {
            return;
        }
        if (typeof window.YT !== 'undefined' && event.data === window.YT.PlayerState.PLAYING) {
            tvStateMachine.transition('PLAY');
            // Clear any existing interval first
            if (tv.progressInterval) {
                clearInterval(tv.progressInterval);
            }
            tv.serverUpdateCounter = 0;

            tv.progressInterval = setInterval(function() {
                if (tv.players && tv.players.youtube &&
                    typeof tv.players.youtube.getCurrentTime === 'function' &&
                    typeof tv.players.youtube.getDuration === 'function' &&
                    tv.players.youtube.getPlayerState() === 1) {  // Only update if playing
                    const currentTime = tv.players.youtube.getCurrentTime();
                    const duration = tv.players.youtube.getDuration();
                    if (duration > 0) {
                        const progress = (currentTime / duration) * 100;
                        const roundedProgress = Math.round(progress);

                        // Update local state immediately for responsive UI
                        if (updateProgress) {
                            updateProgress(currentMediaIdRef.current, roundedProgress);
                        }
                        tv.serverUpdateCounter++;
                        if (tv.serverUpdateCounter % 10 === 0) {
                            fetch(window.baseUrl + '/tv/progress', {
                                method: 'POST',
                                headers: {'Content-Type': 'application/json'},
                                body: JSON.stringify({media_id: currentMediaIdRef.current, progress: roundedProgress})
                            }).then(() => {
                            }).catch(e => console.error('Progress update failed:', e));
                        }
                    }
                }
            }, 1000);
        } else if (typeof window.YT !== 'undefined' && event.data === window.YT.PlayerState.PAUSED) {
            tvStateMachine.transition('PAUSE');
            if (tv.progressInterval) {
                clearInterval(tv.progressInterval);
                tv.progressInterval = null;
            }
        } else if (typeof window.YT !== 'undefined' && event.data === window.YT.PlayerState.ENDED) {
            if (tv.progressInterval) {
                clearInterval(tv.progressInterval);
                tv.progressInterval = null;
            }
            // Check if there is a next eligible item using the latest playlist from props
            const nextIdx = tvStateMachine.getNextQualifyingItem(params.channel.playlist);
            if (typeof nextIdx === 'number' && nextIdx !== -1) {
                if (onVideoEnd) {
                    onVideoEnd(); // This triggers NEXT_ITEM
                }
            } else {
                tvStateMachine.transition('END');
            }
        }
    }, [updateProgress, onVideoEnd, params.channel.playlist]);

    const loadVideo = useCallback((item) => {
        const tv = document.querySelector('.Tv');
        const videoId = getVideoId(item.url);
        const durationSeconds = typeof item.duration === 'number' ? item.duration : parseDuration(item.duration);
        const startSeconds = item.progress > 10 ? Math.round((item.progress / 100) * durationSeconds) : getInSeconds(item.startTs);
        tv.players.youtube.loadVideoById(videoId, startSeconds);
        currentMediaIdRef.current = item.id;
    }, [currentMediaIdRef]);

    useEffect(() => {
        // Load the YouTube API if not already loaded.
        if (!document.getElementById('iframe-demo')) {
            let tag = document.createElement('script');
            tag.id = 'iframe-demo';
            tag.src = 'https://www.youtube.com/iframe_api';
            let firstScriptTag = document.getElementsByTagName('script')[0];
            firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
        }
    }, []);

    // eslint-disable-next-line react-hooks/exhaustive-deps

    useEffect(() => {
        function initializePlayer() {
            const tv = tvRef.current;
            if (!tv) return;
            // Only create if not already exists and YT is available
            if (!tv.players && typeof window.YT !== 'undefined') {
                try {
                    tv.players = tv.players || {};
                    tv.players.youtube = new window.YT.Player(tv, {
                        height: '100%',
                        width: '100%',
                        videoId: getVideoId(videoRef.current.url),
                        playerVars: {
                            rel: 0,
                            modestbranding: 1,
                            fs: 0,
                            disablekb: 1,
                            autoplay: 1,
                            mute: 1,
                            cc_load_policy: channelRef.current.closedCaptions ?? 0,
                            iv_load_policy: channelRef.current.annotations ?? 3,
                            enablejsapi: 1,
                            controls: 0,
                        },
                        events: {
                            onReady: function(event) {
                                // Only seek if this is the intended video
                                const playerVideoId = event.target && event.target.getVideoData && event.target.getVideoData().video_id;
                                if (playerVideoId && playerVideoId === currentVideoIdRef.current) {
                                    const startSeconds = videoRef.current.progress > 10 ? Math.round((videoRef.current.progress / 100) * durationSecondsRef.current) : getInSeconds(videoRef.current.startTs);
                                    if (startSeconds > 0) {
                                        event.target.seekTo(startSeconds);
                                    }
                                    // Removed auto-play to avoid browser blocking; user must click play
                                    if (typeof handleStateChange === 'function') handleStateChange(event, 'ready');
                                }
                            },
                            onStateChange: (event) => handleStateChange(event, tv),
                        },
                    });
                    // Provide standardized player interface.
                    tv.players.youtube.play = function () {
                        tv.players.youtube.playVideo();
                    };
                    tv.players.youtube.pause = function () {
                        tv.players.youtube.pauseVideo();
                    };
                    tv.players.youtube.load = loadVideo;
                    tv.players.youtube.playPause = function () {
                        if (tv.players.youtube.getPlayerState() === 1) {
                            tv.players.youtube.pause();
                        } else {
                            tv.players.youtube.play();
                        }
                    };
                    // Set the default player to be the YouTube player.
                    tv.defaultPlayer = tv.players.youtube;
                } catch (e) {
                    console.error('Player creation failed:', e);
                }
            }
        }
        // Set up API ready handler only if not already set
        if (!window.youtubeApiReady) {
            window.onYouTubeIframeAPIReady = function () {
                window.youtubeApiReady = true;
                initializePlayer();
            };
        } else {
            // API already ready, initialize immediately
            initializePlayer();
        }
    }, [handleStateChange, loadVideo, tvRef]);


    if (!item.id) {
        return <div>No video at this index</div>;
    }

    if (!item.url || !item.url.includes('youtube.com')) {
        return <div>Not a YouTube video</div>;
    }

    return (
        <div style={{position: 'relative', width: '100%', height: '100%'}}></div>
    );
}

export default YouTube;
