import React, { useState, useRef, useEffect, useCallback } from 'react';
import tvStateMachine from './tvStateMachine';
import Keys from './Components/Keys';
import './Tv.css';
import Player from "./Components/Player/Player";
import { useChannelsInit } from "./Components/ChannelList";
import { useChannelApi, useChannelStateSync, handleResetProgress as handleResetProgressFn, useChannelFetch } from "./Components/ChannelItems";
import Accordion from "./Components/Accordion";
import ChannelList from './Components/ChannelList';
import ChannelItems from './Components/ChannelItems';
import Controls from './Components/Controls';
import Mouse from './Components/Mouse';
import { updateProgress as updateProgressFn } from './Components/Player/playbackEvents';
import { getTvCookieName, setCookie } from './helpers';

function TvContainer(params) {
    // Lower third overlay state
    const [showBrandOverlay, setShowBrandOverlay] = useState(false);

    // Subscribe to playerState changes from tvStateMachine
    React.useEffect(() => {
        const unsubscribe = tvStateMachine.subscribe((event, prev, next) => {
            if (prev.playerState !== next.playerState) {
                setPlayerState(next.playerState);
            }
        });
        return () => unsubscribe();
    }, []);

    // Sync React fullscreen state with browser fullscreen API (handles Esc key exit)
    React.useEffect(() => {
        function handleFullscreenChange() {
            const isFullscreen = document.fullscreenElement || document.webkitFullscreenElement || document.mozFullScreenElement || document.msFullscreenElement;
            if (!isFullscreen) {
                setFullscreen(false);
                tvStateMachine.transition('FULLSCREEN_EXIT');
            }
        }
        document.addEventListener('fullscreenchange', handleFullscreenChange);
        document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
        document.addEventListener('mozfullscreenchange', handleFullscreenChange);
        document.addEventListener('MSFullscreenChange', handleFullscreenChange);
        return () => {
            document.removeEventListener('fullscreenchange', handleFullscreenChange);
            document.removeEventListener('webkitfullscreenchange', handleFullscreenChange);
            document.removeEventListener('mozfullscreenchange', handleFullscreenChange);
            document.removeEventListener('MSFullscreenChange', handleFullscreenChange);
        };
    }, []);

    const tvRef = useRef(null);
    const prevChannelIdRef = useRef(null);
    const [autoHeight, setAutoHeight] = useState(700);
    const [isHovered, setIsHovered] = useState(false);
    const [isIdle, setIsIdle] = useState(false);
    const [playerState, setPlayerState] = useState(tvStateMachine.getState().playerState);
    const [channel, setChannel] = useState({ title: '', playlist: [] });
    const [channels, setChannels] = useState([]);
    const [currentItem, setCurrentItem] = useState(0);
    const [currentChannelId, setCurrentChannelId] = useState(null);
    const [openIndex, setOpenIndex] = useState(() => channels.length <= 1 ? 1 : 0); // Channel menu if 1 or fewer channels, Channels menu if multiple
    const [currentFocusIndex, setCurrentFocusIndex] = useState(0);

    // Refs for accordion headings
    const channelsHeadingRef = useRef(null);
    const channelHeadingRef = useRef(null);

    // Helper function to determine if an element should be focused based on nav map
    const getFocusState = (section, index = null, isHeading = false) => {
        let navIndex = 0;

        // Channels heading
        if (section === 'channels' && isHeading) return currentFocusIndex === 0;
        navIndex++;

        // Channels items (if open)
        if (openIndex === 0) {
            if (section === 'channels' && !isHeading) {
                return currentFocusIndex === navIndex + index;
            }
            navIndex += channels.length;
        }

        // Channel heading
        if (section === 'channel' && isHeading) return currentFocusIndex === navIndex;
        navIndex++;

        // Channel items (if open)
        if (openIndex === 1) {
            if (section === 'channel' && !isHeading) {
                return currentFocusIndex === navIndex + index;
            }
            navIndex += (channel.playlist || []).length;
        }

        // Controls (last 3 items)
        if (section === 'controls') {
            return currentFocusIndex === navIndex + index;
        }

        return false;
    };

    // Helper function to get the focused control index
    const getFocusedControlIndex = () => {
        if (getFocusState('controls', 0)) return 0;
        if (getFocusState('controls', 1)) return 1;
        if (getFocusState('controls', 2)) return 2;
        if (getFocusState('controls', 3)) return 3;
        return -1;
    };

    const [uiActive] = useState(true);
    const baseUrl = params.baseUrl || '';
    // Set window.baseUrl for progress updates
    if (typeof window !== 'undefined') {
        window.baseUrl = baseUrl;
    }
    const [fullscreen, setFullscreen] = useState(false);

    const { tvWidth, playerHeight, tvContainerRef } = useTvResize();
    const { getChannel, getChannels } = useChannelApi(baseUrl, currentChannelId, setChannel, setChannels);
    useChannelsInit(getChannels, channels, currentChannelId, setCurrentChannelId);
    useChannelFetch(currentChannelId, getChannel);
    useChannelStateSync(channel, setCurrentItem, setCurrentFocusIndex, openIndex);

    // Custom item selection handler that manages focus
    const onItemSelect = useCallback((item) => {
        let index = -1;
        if (typeof item === 'number') {
            index = item;
        } else {
            index = channel.playlist ? channel.playlist.findIndex(playlistItem => playlistItem.id === item.id) : -1;
        }
        if (index !== -1) {
            tvStateMachine.transition('PLAY');
            setCurrentItem(index);
            // Set focus to the selected item for keyboard navigation
            // Focus indices: 0=channels heading, 1=channel heading, 2+=channel items
            setCurrentFocusIndex(2 + index);
        }
    }, [channel.playlist, setCurrentItem, setCurrentFocusIndex]);

    const updateProgress = updateProgressFn(channel, setChannel);
    const handleResetProgress = handleResetProgressFn(channel, updateProgress, baseUrl, currentItem, tvRef);
    // Auto-advance to next item when video ends
    const handleVideoEnd = () => {
        tvStateMachine.transition('NEXT_ITEM', { playlist: channel.playlist });
    };
    const handleVideoLoad = () => {};

    // Custom channel selection handler that sets the first qualifying item
    const onChannelSelect = useCallback((channelId) => {
        // Only switch if it's a different channel
        if (String(channelId) === String(currentChannelId)) {
            return;
        }
        setCurrentChannelId(channelId);
        // Save channel choice to cookie (unique per page path)
        const cookieName = getTvCookieName();
        setCookie(cookieName, channelId);
        // The first qualifying item will be set when the new channel data loads
        // via useChannelStateSync - don't call PLAY here as it causes race conditions
    }, [currentChannelId, setCurrentChannelId]);

    useFullscreenResize(tvRef, setFullscreen);

    // Dynamically set .Tv height based on content (player + controls + sidebar)
    React.useLayoutEffect(() => {
        // Only use 100vh if fullscreen and not paused
        if (fullscreen && playerState !== 'paused') {
            setAutoHeight('100vh');
        } else {
            setAutoHeight(700);
        }
    }, [fullscreen, playerState]);

    // Subscribe to state machine events
    React.useEffect(() => {
        const unsubscribe = tvStateMachine.subscribe((event) => {
            if (event === 'RESET_AND_START') {
                // Reset all progress
                setChannel(prevChannel => {
                    const updatedPlaylist = prevChannel.playlist.map(item => {
                        // Send to server
                        fetch(baseUrl + '/tv/progress', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify({media_id: item.id, progress: 0})
                        }).catch(e => console.error('Progress reset failed:', e));
                        return { ...item, progress: 0 };
                    });
                    return { ...prevChannel, playlist: updatedPlaylist };
                });
                // Start playing from first item
                tvStateMachine.transition('PLAY');
            }
        });
        return unsubscribe;
    }, [setChannel, baseUrl]);

    // Check if all items are completed on channel load (only when channel ID changes)
    React.useEffect(() => {
        // Only run this check when we've actually switched to a new channel
        if (!currentChannelId || prevChannelIdRef.current === currentChannelId) {
            return;
        }
        // Update the ref to track this channel
        prevChannelIdRef.current = currentChannelId;

        if (channel && channel.playlist && channel.playlist.length > 0) {
            const allCompleted = channel.playlist.every(item => {
                const duration = parseFloat(item.duration) || 0;
                const progress = parseFloat(item.progress) || 0;
                const remaining = duration - (progress / 100 * duration);
                return remaining < 30;
            });
            if (allCompleted) {
                setChannel(prevChannel => {
                    const updatedPlaylist = prevChannel.playlist.map(item => {
                        fetch(baseUrl + '/tv/progress', {
                            method: 'POST',
                            headers: {'Content-Type': 'application/json'},
                            body: JSON.stringify({media_id: item.id, progress: 0})
                        }).catch(e => console.error('Progress reset failed:', e));
                        return { ...item, progress: 0 };
                    });
                    return { ...prevChannel, playlist: updatedPlaylist };
                });
                setCurrentItem(0);
                tvStateMachine.transition('PLAY');
            }
        }
    }, [currentChannelId, channel, baseUrl, setChannel, setCurrentItem]);

    // Update openIndex based on number of channels
    React.useEffect(() => {
        if (channels.length > 1) {
            // Open Channels menu
            setOpenIndex(0);
        } else if (channels.length === 1) {
            // Open Channel menu
            setOpenIndex(1);
        }
    }, [channels.length]);

    // Focus the main container for keyboard navigation
    React.useEffect(() => {
        const focusContainer = () => {
            if (tvContainerRef.current) {
                tvContainerRef.current.focus();
            }
        };

        // Focus immediately
        focusContainer();

        // Also focus after a short delay to handle fullscreen transitions
        const timeoutId = setTimeout(focusContainer, 100);

        return () => clearTimeout(timeoutId);
    }, [fullscreen, tvContainerRef]);

    // Hide controls/sidebar and expand player if fullscreen & idle
    const hideUi = fullscreen && isIdle;

    // Brand string from channel data (set per-channel or via default brand provider)
    const brandString = channel.brand || '';

    // Refs for brand overlay timers
    const brandShowTimerRef = useRef(null);
    const brandHideTimerRef = useRef(null);

    // Helper to clear brand overlay timers
    const clearBrandTimers = useCallback(() => {
        if (brandShowTimerRef.current) {
            clearTimeout(brandShowTimerRef.current);
            brandShowTimerRef.current = null;
        }
        if (brandHideTimerRef.current) {
            clearTimeout(brandHideTimerRef.current);
            brandHideTimerRef.current = null;
        }
        setShowBrandOverlay(false);
    }, []);

    // Subscribe to PLAY/PAUSE events to show brand overlay (60s after play, show for 10s)
    React.useEffect(() => {
        // Only subscribe if brand string is provided
        if (!brandString) return;

        // Subscribe to state machine events
        const unsubscribe = tvStateMachine.subscribe((event, prev, next) => {
            const startedPlaying = (event === 'PLAY' || event === 'NEXT_ITEM') || 
                (prev && next && prev.playerState !== 'playing' && next.playerState === 'playing');
            const stoppedPlaying = event === 'PAUSE' || event === 'SET_CURRENT_ITEM' ||
                (prev && next && prev.playerState === 'playing' && next.playerState !== 'playing');

            if (startedPlaying) {
                // Clear any existing timers first
                clearBrandTimers();
                // Show brand overlay 60 seconds after video starts
                brandShowTimerRef.current = setTimeout(() => {
                    setShowBrandOverlay(true);
                    // Hide after 30 seconds
                    brandHideTimerRef.current = setTimeout(() => {
                        setShowBrandOverlay(false);
                    }, 30000);
                }, 60000);
            }

            if (stoppedPlaying) {
                // Clear timers and hide overlay on pause or video change
                clearBrandTimers();
            }
        });

        return () => {
            unsubscribe();
            clearBrandTimers();
        };
    }, [brandString, clearBrandTimers]);

    // Handle keyboard events on the main container
    const handleKeyDown = React.useCallback((e) => {
        // Prevent default browser focus behavior for Tab to avoid focusing player
        if (e.key === 'Tab' || (e.key === 'Tab' && e.shiftKey)) {
            e.preventDefault();
        }
    }, []);

    return (
        !channel || typeof channel.title === 'undefined' ? null : (
            <div
                className={`Tv${uiActive ? ' Tv--ui-active' : ' Tv--ui-idle'}${fullscreen ? ' fullscreen' : ''}${hideUi ? ' tv-hide-ui' : ''}`}
                ref={tvContainerRef}
                tabIndex={0}
                data-tv-state={JSON.stringify(tvStateMachine.getState())}
                style={{ width: '100%', maxWidth: '100vw', height: autoHeight, display: 'flex', flexDirection: 'row', overflow: 'hidden' }}
                onKeyDown={handleKeyDown}
            >
                {/* Mouse activity tracker for idle/active UI state */}
                <Mouse
                    tvRef={tvContainerRef}
                    isHovered={isHovered}
                    setIsHovered={setIsHovered}
                    isIdle={isIdle}
                    setIsIdle={setIsIdle}
                />
                <Keys
                    channels={channels}
                    channel={channel}
                    setCurrentChannelId={setCurrentChannelId}
                    setCurrentItem={setCurrentItem}
                    setOpenIndex={setOpenIndex}
                    openIndex={openIndex}
                    currentFocusIndex={currentFocusIndex}
                    setCurrentFocusIndex={setCurrentFocusIndex}
                    tvRef={tvRef}
                    setIsIdle={setIsIdle}
                />
                <div style={{ display: 'flex', flex: 1, flexDirection: 'row', minWidth: 0, height: '100%' }}>
                    <div
                        className="tv-main-col"
                        style={{
                            width: `${tvWidth}px`,
                            minWidth: `${tvWidth}px`,
                            maxWidth: `${tvWidth}px`,
                            height: '100%',
                            flex: '0 0 auto',
                            ...(hideUi ? { width: '100vw', minWidth: 0, maxWidth: '100vw', zIndex: 10 } : {})
                        }}
                    >
                        <div
                            className="tv-player-container"
                            style={{
                                height: hideUi ? '100vh' : `${playerHeight}px`,
                                minHeight: hideUi ? '100vh' : `${playerHeight}px`,
                                maxHeight: hideUi ? '100vh' : `${playerHeight}px`,
                                width: hideUi ? '100vw' : undefined,
                                zIndex: hideUi ? 11 : undefined,
                                overflow: 'hidden',
                            }}
                            tabIndex={-1}
                        >
                            <Player
                                channel={channel}
                                item={currentItem}
                                onVideoEnd={handleVideoEnd}
                                updateProgress={updateProgress}
                                onVideoLoad={handleVideoLoad}
                                tvRef={tvRef}
                            />
                            {brandString && (
                                <div className={`tv-brand-overlay${showBrandOverlay ? ' tv-brand-overlay--visible' : ''}`}>
                                    {brandString}
                                </div>
                            )}
                        </div>
                        {!fullscreen && !hideUi && (
                            <div className="tv-controls-bar">
                                <Controls tvRef={tvRef} containerRef={tvContainerRef} playerState={playerState} fullscreen={fullscreen} controlsFocusIdx={getFocusedControlIndex()} />
                            </div>
                        )}
                        {fullscreen && !hideUi && (
                            <div className="tv-controls-float">
                                <Controls tvRef={tvRef} containerRef={tvContainerRef} playerState={playerState} size="large" fullscreen={fullscreen} controlsFocusIdx={getFocusedControlIndex()} />
                            </div>
                        )}
                    </div>
                    {/* Sidebar right: controls and navigation */}
                    {!hideUi && (
                        <div
                            className={`tv-sidebar${uiActive ? '' : ' tv-sidebar-hidden'}`}
                            style={{ minWidth: 240, maxWidth: 340, width: '22vw', flex: '0 0 22vw', transition: 'opacity 0.7s', opacity: uiActive ? 1 : 0, pointerEvents: uiActive ? 'auto' : 'none', background: '#181818', borderLeft: '1px solid #222', height: '100%' }}
                        >
                            <Accordion openIndex={openIndex} setOpenIndex={setOpenIndex}>
                                <ChannelList
                                    channels={channels}
                                    currentChannelId={currentChannelId}
                                    onChannelSelect={onChannelSelect}
                                    focused={getFocusState('channels', null, true)}
                                    headingRef={channelsHeadingRef}
                                    getFocusState={getFocusState}
                                />
                                <ChannelItems
                                    channel={channel}
                                    currentItem={currentItem}
                                    onItemSelect={onItemSelect}
                                    handleResetProgress={handleResetProgress}
                                    focused={getFocusState('channel', null, true)}
                                    headingRef={channelHeadingRef}
                                    getFocusState={getFocusState}
                                />
                            </Accordion>
                        </div>
                    )}
                </div>
            </div>
        )
    );
}

// Fullscreen monitoring and sync hook
function useFullscreenResize(tvRef, setFullscreen) {
    useEffect(() => {
        function handleFullscreenChange() {
            const isFs = document.fullscreenElement || document.webkitFullscreenElement || document.msFullscreenElement;
            setFullscreen(!!isFs);
        }
        document.addEventListener('fullscreenchange', handleFullscreenChange);
        document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
        document.addEventListener('msfullscreenchange', handleFullscreenChange);
        // Also subscribe to tvStateMachine fullscreen state
        let unsub = null;
        function syncFullscreenState(next) {
            try {
                if (typeof next.fullscreen === 'boolean') setFullscreen(!!next.fullscreen);
            } catch (e) {
                console.error('Error syncing fullscreen state:', e);
            }
        }
        try {
            unsub = tvStateMachine.subscribe((_, __, next) => syncFullscreenState(next));
        } catch (e) {
            console.error('Error subscribing to fullscreen state:', e);
            unsub = () => {};
        }
        return () => {
            document.removeEventListener('fullscreenchange', handleFullscreenChange);
            document.removeEventListener('webkitfullscreenchange', handleFullscreenChange);
            document.removeEventListener('msfullscreenchange', handleFullscreenChange);
            if (unsub) unsub();
        };
    }, [tvRef, setFullscreen]);
}

function useTvResize() {
    const [tvWidth, setTvWidth] = useState(0);
    const [controlsHeight, setControlsHeight] = useState(0);
    const tvContainerRef = useRef(null);

    useEffect(() => {
        function handleResize() {
            if (tvContainerRef.current) {
                // Try to find sidebar and controls elements
                const sidebar = tvContainerRef.current.parentElement?.querySelector('.tv-sidebar');
                const controls = tvContainerRef.current.querySelector('.tv-controls-bar');
                // Enforce min sidebar width and controls height
                const minSidebar = 200;
                const minControls = 60;
                const actualSidebar = sidebar ? Math.max(sidebar.offsetWidth, minSidebar) : minSidebar;
                const actualControls = controls ? Math.max(controls.offsetHeight, minControls) : minControls;
                setControlsHeight(actualControls);
                setTvWidth(tvContainerRef.current.offsetWidth - actualSidebar);
            }
        }
        handleResize();
        window.addEventListener('resize', handleResize);
        return () => window.removeEventListener('resize', handleResize);
    }, []);

    const playerHeight = Math.round(tvWidth / (16 / 9));
    const tvHeight = playerHeight + controlsHeight;

    return { tvWidth, tvHeight, playerHeight, tvContainerRef };
}

export default TvContainer;
