import React from 'react';
import ReactDOM from 'react-dom/client';
import './index.css';
import TvContainer from './TvContainer';
import ErrorBoundary from './Components/ErrorBoundary';
import reportWebVitals from './reportWebVitals';

const MAX_RETRY_ATTEMPTS = 10;
const RETRY_INTERVAL_MS = 200;

/**
 * Attempts to mount the TV React app to .tv elements.
 * Returns true if at least one element was found and mounted.
 */
function tryMountTv() {
  const tvs = document.getElementsByClassName('tv');
  if (tvs.length === 0) {
    return false;
  }

  for (let tv of tvs) {
    // Skip if already mounted
    if (tv.dataset.tvMounted === 'true') {
      continue;
    }

    let baseUrl = tv.dataset.baseUrl || '';
    let channel = tv.dataset.channel || 1;

    ReactDOM.createRoot(tv).render(
      <React.StrictMode>
        <ErrorBoundary>
          <TvContainer root={tv} baseUrl={baseUrl} channel={channel} />
        </ErrorBoundary>
      </React.StrictMode>
    );

    // Mark as mounted to prevent double-mounting
    tv.dataset.tvMounted = 'true';
  }

  return true;
}

/**
 * Robust initialization that handles various loading scenarios:
 * 1. DOM already ready (script loaded async after DOMContentLoaded)
 * 2. DOM not ready yet (script in head or sync loaded)
 * 3. Elements added dynamically after page load
 */
function initTv() {
  // Try immediately first (handles async script loaded after DOM ready)
  if (tryMountTv()) {
    console.log('TV app initialized immediately');
    return;
  }

  // If DOM isn't ready yet, wait for it
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function () {
      if (tryMountTv()) {
        return;
      }
      // If still not found, start polling
      pollForTvElements();
    });
  } else {
    // DOM is ready but no elements found - poll for dynamically added elements
    pollForTvElements();
  }
}

/**
 * Poll for .tv elements that might be added dynamically.
 * Falls back to MutationObserver if available.
 */
function pollForTvElements() {
  let attempts = 0;

  // Try using MutationObserver for efficiency
  if (typeof MutationObserver !== 'undefined') {
    const observer = new MutationObserver((mutations, obs) => {
      if (tryMountTv()) {
        obs.disconnect();
      }
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true
    });

    // Also set a timeout to stop observing after max attempts equivalent time
    setTimeout(() => {
      observer.disconnect();
      if (!document.querySelector('.tv[data-tv-mounted="true"]')) {
        console.warn('TV app: No .tv elements found after waiting. The TV block may not be present on this page.');
      }
    }, MAX_RETRY_ATTEMPTS * RETRY_INTERVAL_MS);

    return;
  }

  // Fallback: interval-based polling
  const pollInterval = setInterval(() => {
    attempts++;

    if (tryMountTv()) {
      clearInterval(pollInterval);
      return;
    }

    if (attempts >= MAX_RETRY_ATTEMPTS) {
      clearInterval(pollInterval);
      console.warn('TV app: No .tv elements found after maximum retry attempts. The TV block may not be present on this page.');
    }
  }, RETRY_INTERVAL_MS);
}

// Initialize the TV app
initTv();

// If you want to start measuring performance in your app, pass a function
// to log results (for example: reportWebVitals(console.log))
// or send to an analytics endpoint. Learn more: https://bit.ly/CRA-vitals
reportWebVitals();
