<?php

namespace Drupal\tv\Service;

use DateTime;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\Entity\File;
use Drupal\image\Entity\ImageStyle;
use Drupal\media\Entity\Media;
use Drupal\node\NodeInterface;
use Drupal\tv\Entity\TvChannel;
use Drupal\tv\Timestamp;
use Drupal\user\UserDataInterface;

class TvChannelService implements TvChannelServiceInterface {

    private UserDataInterface $userData;
    private AccountProxyInterface $user;
    private ModuleHandlerInterface $moduleHandler;

    public function __construct(AccountProxyInterface $user, UserDataInterface $userData) {
        $this->user = $user;
        $this->userData = $userData;
        $this->moduleHandler = \Drupal::service('module_handler');
    }

    public function getItemIds(TvChannel $channel): array
    {
        $channelItems = [];
        $tags = $channel->getTags();
        if (empty($tags)) {
            return $channelItems;
        }

        // Select videos with having all the tags the channel has.
        // @todo add support for `video` bundle: $bundles[] = 'video';
        $bundles[] = 'remote_video';
        $bundles[] = 'clip';
        $query = \Drupal::entityQuery('media')
            ->condition('bundle', $bundles, 'IN')
            ->condition('field_duration', 0, '>')
            ->condition('field_description', NULL, 'IS NOT NULL')
            ->condition('field_thumbnail', NULL, 'IS NOT NULL')
            ->condition('status', 1)
            ->sort('field_weight')
            ->sort('created', 'DESC')
            ->accessCheck();

        // Each video must have all the tags the channel has.
        foreach ($tags as $tag) {
            $query->condition($query->andConditionGroup()
                ->condition('field_tags', $tag->id()));
        }

        // Allow other modules to modify query.
        $this->moduleHandler->alter('tv_channel_items_query', $query, $channel);

        return $query->execute();
    }

    public function getItems(TvChannel $node): array
    {
        $channelItems = [];
        $mediaIds = $this->getItemIds($node);

        // @todo Move videos the user has watched to the end of the playlist.
        // $completedVideoIds = $this->getCompletedVideoIds();

        // Create the playlist.
        foreach ($mediaIds as $mid) {
            $video = Media::load($mid);
            $duration = $video->get('field_duration')->first()?->getValue()['value'];
            $duration = is_numeric($duration) && $duration > 0 ? (int)$duration : 90;
            $item = [
                'id' => $video->id(),
                'name' => $video->getName(),
                'url' => $video->get('field_media_oembed_video')->first()->getValue()['value'],
                'createdDt' => (new DateTime())->setTimestamp($video->getCreatedTime())->format('c'),
                'startedDt' => $this->getStartedDateTime($video)?->format('c'),
                'startTs' => $this->getStartTimestamp($video),
                'endTs' => $this->getEndTimestamp($video),
                'duration' => $duration,
                'progress' => $this->getProgress($video->id()),
            ];

            $image_media = $video->get('field_thumbnail')->entity ?? null;
            if ($image_media) {
              $managed_image_id = $image_media->get('field_media_image')->first()->getValue()['target_id'];
              $imageUri = File::load($managed_image_id)->getFileUri();

              $style = ImageStyle::load('thumbnail');
              $derivativeUrl = $style->buildUrl($imageUri);

              $item['posterUrl'] = $derivativeUrl;
            }

            $channelItems[] = $item;
        }

        // Allow other modules to modify the items.
        $this->moduleHandler->alter('tv_channel_items', $channelItems, $node);

        return $channelItems;
    }

    private function getStartedDateTime(Media $video): ?\DateTime
    {
        // @todo return when the user started watching the video.
        return NULL;
    }

    private function getCompletedDateTime(Media $video): ?\DateTime
    {
        // @todo return when the user completed watching the video.
        return NULL;
    }

    private function getStartTimestamp(Media $video): ?string
    {
        // @todo Implement getStartTimestamp() method.
        return "00:00:00";
    }

    public function updateProgress(int $mediaId, float $progress): void
    {
        if ($this->user->id() == 0) {
            // Anonymous user: save in session
            $session = \Drupal::request()->getSession();
            $key = 'tv_progress_' . $mediaId;
            $session->set($key, $progress);
        } else {
            $this->userData->set('tv', $this->user->id(), 'progress_' . $mediaId, $progress);
        }
    }

    private function getProgress(int $mediaId): float
    {
        if ($this->user->id() == 0) {
            // Anonymous user: load from session
            $session = \Drupal::request()->getSession();
            $key = 'tv_progress_' . $mediaId;
            return round($session->get($key, 0));
        } else {
            return round($this->userData->get('tv', $this->user->id(), 'progress_' . $mediaId) ?? 0);
        }
    }

    private function getEndTimestamp(Media $video): ?string
    {
        // @todo return the timestamp where the video should end.
        return NULL;
    }

    private function getCompletedVideoIds(): array
    {
        // @todo return the ids of the videos the user has completed.
        // $uid = $this->user->id();
        // $userData->set('tv', $uid, 'history', []);
        // $history = $userData->get('tv', $uid, 'history') ?? [];
        return [];
    }

    public function getAutoPlayPreference(): bool
    {
        return $this->userData->get('tv', $this->user->id(), 'autoplay') ?? FALSE;
    }

    public function getMutePreference(): bool
    {
        return $this->userData->get('tv', $this->user->id(), 'mute') ?? TRUE;
    }
}
