/**
 * ErrorBoundary - React error boundary for catching and displaying component errors
 * 
 * @component
 * @scope Error handling wrapper for entire TV app or sub-components
 * @responsibilities
 * - Catch JavaScript errors in child component tree
 * - Display fallback UI with error details when errors occur
 * - Log error information to console
 * - Prevent entire app from crashing on component errors
 * @limits
 * - ❌ Only catches errors in render, lifecycle methods, and constructors
 * - ❌ Does NOT catch errors in event handlers, async code, or SSR
 * - ✅ Should wrap root component or critical sub-trees
 * @see .cursorrules for full architecture boundaries
 * @see https://react.dev/reference/react/Component#catching-rendering-errors-with-an-error-boundary
 */
import React from 'react';

/**
 * ErrorBoundary class component - Catches errors in child components
 * @example
 * <ErrorBoundary>
 *   <TvContainer />
 * </ErrorBoundary>
 */
class ErrorBoundary extends React.Component {
  constructor(props) {
    super(props);
    this.state = { hasError: false, error: null, errorInfo: null };
  }

  static getDerivedStateFromError(error) {
    // Update state so the next render will show the fallback UI
    return { hasError: true };
  }

  componentDidCatch(error, errorInfo) {
    // Log the error details
    console.error('ErrorBoundary caught an error:', error, errorInfo);
    this.setState({
      error: error,
      errorInfo: errorInfo
    });
  }

  render() {
    if (this.state.hasError) {
      // Render error UI
      return (
        <div style={{
          padding: '20px',
          backgroundColor: '#f5f5f5',
          border: '1px solid #ccc',
          borderRadius: '5px',
          fontFamily: 'monospace',
          color: '#333'
        }}>
          <h2>Something went wrong</h2>
          <details style={{ whiteSpace: 'pre-wrap' }}>
            <summary>Error Details</summary>
            <p><strong>Error:</strong> {this.state.error && this.state.error.toString()}</p>
            <p><strong>Stack:</strong></p>
            <pre>{this.state.error && this.state.error.stack}</pre>
            <p><strong>Component Stack:</strong></p>
            <pre>{this.state.errorInfo && this.state.errorInfo.componentStack}</pre>
          </details>
        </div>
      );
    }

    return this.props.children;
  }
}

export default ErrorBoundary;
