/**
 * Mouse - Mouse and keyboard activity tracker for idle/active UI state
 * 
 * @component
 * @scope Tracks mouse hover and idle states for UI visibility (controls, overlays)
 * @responsibilities
 * - Track mouse enter/leave on TV container
 * - Track mouse movement for idle timer
 * - Track keyboard activity for idle timer
 * - Set idle state after 5 seconds of inactivity
 * - Reset idle timer on any user activity
 * @limits
 * - ❌ Must NOT handle keyboard shortcuts (Keys.js handles this)
 * - ❌ Must NOT control player or UI directly
 * - ✅ Should ONLY track activity and update hover/idle state
 * @see .cursorrules for full architecture boundaries
 */
import { useEffect } from 'react';

/**
 * Mouse component - Activity tracker for hover and idle states
 * @param {Object} props - Component props
 * @param {Object} props.tvRef - Ref to main TV container element
 * @param {boolean} props.isHovered - Current hover state
 * @param {Function} props.setIsHovered - Function to update hover state
 * @param {boolean} props.isIdle - Current idle state
 * @param {Function} props.setIsIdle - Function to update idle state (true after 5s inactivity)
 */
function Mouse({ tvRef, isHovered, setIsHovered, isIdle, setIsIdle }) {
  useEffect(() => {
    const tv = tvRef.current;
    if (!tv) return;
    let idleTimeout = null;

    const resetIdleTimer = () => {
      if (isIdle) {
        setIsIdle(false);
      }
      if (idleTimeout) clearTimeout(idleTimeout);
      idleTimeout = setTimeout(() => {
        if (!isIdle) {
          setIsIdle(true);
        }
      }, 5000);
    };

    const handleMouseEnter = () => {
      setIsHovered(true);
    };
    const handleMouseLeave = () => {
      setIsHovered(false);
    };
    const handleMouseMove = (e) => {
      resetIdleTimer();
    };
    const handleKeyDown = (e) => {
      resetIdleTimer();
      if (isHovered) setIsHovered(false);
    };

    tv.addEventListener('mouseenter', handleMouseEnter);
    tv.addEventListener('mouseleave', handleMouseLeave);
    tv.addEventListener('mousemove', handleMouseMove);
    tv.addEventListener('keydown', handleKeyDown);

    idleTimeout = setTimeout(() => {
      if (!isIdle) {
        setIsIdle(true);
      }
    }, 5000);

    return () => {
      tv.removeEventListener('mouseenter', handleMouseEnter);
      tv.removeEventListener('mouseleave', handleMouseLeave);
      tv.removeEventListener('mousemove', handleMouseMove);
      tv.removeEventListener('keydown', handleKeyDown);
      if (idleTimeout) clearTimeout(idleTimeout);
    };
  }, [tvRef, isHovered, setIsHovered, isIdle, setIsIdle]);

  return null;
}

export default Mouse;
