/**
 * Unit tests for tvStateMachine.js
 * Tests state management without React dependencies
 */

import tvStateMachine from '../tvStateMachine';

describe('tvStateMachine', () => {
  beforeEach(() => {
    // Reset state machine before each test
    tvStateMachine.transition('RESET', {
      playerState: 'paused',
      currentItem: 0,
      playlistLength: 0,
      userIntent: null,
      fullscreen: false,
      overlay: null
    });
  });

  describe('initial state', () => {
    test('has default state values', () => {
      const state = tvStateMachine.getState();
      expect(state).toHaveProperty('playerState');
      expect(state).toHaveProperty('currentItem');
      expect(state).toHaveProperty('playlistLength');
      expect(state).toHaveProperty('userIntent');
      expect(state).toHaveProperty('fullscreen');
    });
  });

  describe('state transitions', () => {
    test('updates playerState on PLAY transition', () => {
      tvStateMachine.transition('PLAY');
      const state = tvStateMachine.getState();
      expect(state.playerState).toBe('playing');
    });

    test('updates playerState on PAUSE transition', () => {
      tvStateMachine.transition('PAUSE');
      const state = tvStateMachine.getState();
      expect(state.playerState).toBe('paused');
    });

    test('updates playerState on END transition', () => {
      tvStateMachine.transition('END');
      const state = tvStateMachine.getState();
      expect(state.playerState).toBe('ended');
    });

    test('updates currentItem on SET_CURRENT_ITEM', () => {
      tvStateMachine.transition('SET_CURRENT_ITEM', 5);
      const state = tvStateMachine.getState();
      expect(state.currentItem).toBe(5);
    });

    test('updates playlistLength on SET_PLAYLIST_LENGTH', () => {
      tvStateMachine.transition('SET_PLAYLIST_LENGTH', 10);
      const state = tvStateMachine.getState();
      expect(state.playlistLength).toBe(10);
    });

    test('updates fullscreen on FULLSCREEN_ENTER', () => {
      tvStateMachine.transition('FULLSCREEN_ENTER');
      const state = tvStateMachine.getState();
      expect(state.fullscreen).toBe(true);
    });

    test('updates fullscreen on FULLSCREEN_EXIT', () => {
      tvStateMachine.transition('FULLSCREEN_ENTER');
      tvStateMachine.transition('FULLSCREEN_EXIT');
      const state = tvStateMachine.getState();
      expect(state.fullscreen).toBe(false);
    });

    test('updates userIntent on USER_INTENT', () => {
      tvStateMachine.transition('USER_INTENT', { userIntent: 'play' });
      const state = tvStateMachine.getState();
      expect(state.userIntent).toBe('play');
    });
  });

  describe('subscribe/unsubscribe', () => {
    test('notifies subscriber on state change', () => {
      const callback = jest.fn();
      tvStateMachine.subscribe(callback);

      tvStateMachine.transition('PLAY');

      expect(callback).toHaveBeenCalledWith(
        'PLAY',
        expect.objectContaining({ playerState: expect.any(String) }),
        expect.objectContaining({ playerState: 'playing' })
      );
    });

    test('does not notify after unsubscribe', () => {
      const callback = jest.fn();
      const unsubscribe = tvStateMachine.subscribe(callback);

      unsubscribe();
      tvStateMachine.transition('PLAY');

      expect(callback).not.toHaveBeenCalled();
    });

    test('supports multiple subscribers', () => {
      const callback1 = jest.fn();
      const callback2 = jest.fn();

      tvStateMachine.subscribe(callback1);
      tvStateMachine.subscribe(callback2);

      tvStateMachine.transition('PLAY');

      expect(callback1).toHaveBeenCalled();
      expect(callback2).toHaveBeenCalled();
    });

    test('unsubscribe only affects specific subscriber', () => {
      const callback1 = jest.fn();
      const callback2 = jest.fn();

      const unsubscribe1 = tvStateMachine.subscribe(callback1);
      tvStateMachine.subscribe(callback2);

      unsubscribe1();
      tvStateMachine.transition('PLAY');

      expect(callback1).not.toHaveBeenCalled();
      expect(callback2).toHaveBeenCalled();
    });
  });

  describe('getFirstQualifyingItem', () => {
    test('returns first item with progress < 95% and remaining >= 30s', () => {
      // Mock playlist with various progress states
      const mockPlaylist = [
        { id: 1, progress: 96, startTs: '00:00:00', endTs: '00:01:00' }, // Complete
        { id: 2, progress: 50, startTs: '00:00:00', endTs: '00:02:00' }, // Qualifying
        { id: 3, progress: 10, startTs: '00:00:00', endTs: '00:00:20' }, // < 30s remaining
      ];

      // This is a simplified test - actual implementation may vary
      // The function should skip item 1 (>95%) and item 3 (<30s remaining)
      // and return index 1 (item 2)

      // Note: getFirstQualifyingItem needs to be exported for testing
      // This is a TODO item to make the function testable
    });
  });

  describe('getNextQualifyingItem', () => {
    test('returns next item after current with progress < 95% and remaining >= 30s', () => {
      // TODO: Test next qualifying item logic
      // Need to export function from tvStateMachine.js
    });

    test('wraps around to beginning if no qualifying items after current', () => {
      // TODO: Test wrap-around behavior
    });

    test('returns null if no qualifying items in playlist', () => {
      // TODO: Test all-complete playlist
    });
  });

  describe('getPreviousQualifyingItem', () => {
    test('returns previous item before current with progress < 95% and remaining >= 30s', () => {
      // TODO: Test previous qualifying item logic
      // Need to export function from tvStateMachine.js
    });

    test('returns null if no qualifying items before current', () => {
      // TODO: Test beginning of playlist
    });
  });

  describe('edge cases', () => {
    test('handles transition with undefined data', () => {
      expect(() => {
        tvStateMachine.transition('PLAY');
      }).not.toThrow();
    });

    test('handles invalid transition type gracefully', () => {
      expect(() => {
        tvStateMachine.transition('INVALID_TYPE');
      }).not.toThrow();
    });

    test('getState returns a copy of state (not reference)', () => {
      const state1 = tvStateMachine.getState();
      const state2 = tvStateMachine.getState();

      // Should be different objects
      expect(state1).not.toBe(state2);
      // But with same values
      expect(state1).toEqual(state2);
    });
  });
});
