/**
 * ChannelItems - Channel playlist UI component with progress tracking
 * 
 * @component
 * @scope UI rendering only - displays playlist items with progress bars and handles item selection
 * @limits
 * - ❌ Must NOT handle video playback logic (Player.js handles this)
 * - ❌ Must NOT call YouTube/provider APIs directly (use tvRef methods)
 * - ❌ Must NOT calculate resume positions (Player.js handles this)
 * - ❌ Must NOT modify tvStateMachine state (read-only except via exported hooks)
 * @see .cursorrules for full architecture boundaries
 */
import React from 'react';
import { useEffect, useCallback, useRef } from 'react';
import './Channel.css';
import './ChannelItems.css';
import tvStateMachine from '../tvStateMachine';
import AccordionMenu from './AccordionMenu';
import { formatDuration } from '../helpers';

/**
 * Custom hook to fetch channel when currentChannelId changes
 * @param {string} currentChannelId - The ID of the current channel
 * @param {Function} getChannel - Function to fetch channel data
 */
export function useChannelFetch(currentChannelId, getChannel) {
    useEffect(() => {
        if (currentChannelId) {
            getChannel();
        }
    }, [currentChannelId, getChannel]);
}

/**
 * Handler factory to reset progress for an item
 * Resets localStorage, updates progress map, and seeks to startTs if currently playing
 * @param {Object} channel - Channel object
 * @param {Function} updateProgress - Function to update progress in parent
 * @param {string} baseUrl - Base URL for API calls
 * @param {number} currentItem - Index of currently playing item
 * @param {Object} tvRef - Reference to TV player
 * @param {Function} setProgressMap - Function to update progress map state
 * @returns {Function} Handler function (item, index) => void
 */
export const handleResetProgress = (channel, updateProgress, baseUrl, currentItem, tvRef, setProgressMap) => (item, index) => {
    if (typeof item === 'object' && typeof index === 'number' && item.id) {
        // Reset progress in localStorage
        try {
            localStorage.setItem('tv_progress_' + item.id, '0');
        } catch (e) {}
        updateProgress(item.id, 0);
        // Update local progressMap state if provided
        if (typeof setProgressMap === 'function') {
            setProgressMap(prev => ({ ...prev, [item.id]: 0 }));
        }
        // If this is the currently playing item, seek to startTs
        if (index === currentItem && tvRef.current && tvRef.current.players && tvRef.current.players.youtube && typeof tvRef.current.players.youtube.seekTo === 'function') {
            let startSeconds = 0;
            if (item.startTs) {
                const parts = item.startTs.split(':').map(parseFloat);
                if (parts.length >= 3) {
                    startSeconds = parts[0] * 3600 + parts[1] * 60 + parts[2];
                }
            }
            tvRef.current.players.youtube.seekTo(startSeconds);
        }
    }
};

/**
 * Custom hook to sync playlist length, currentItem, and handle user selection
 * Manages auto-play on channel switch and subscribes to tvStateMachine for currentItem updates
 * @param {Object} channel - Channel object with playlist
 * @param {Function} setCurrentItem - Function to update current item state
 * @param {Function} setCurrentFocusIndex - Function to update focus index
 * @param {number} openIndex - Currently open accordion index
 * @param {number} currentItem - Currently playing item index
 * @returns {Object} Object with markUserSelected function
 */
export function useChannelStateSync(channel, setCurrentItem, setCurrentFocusIndex, openIndex, currentItem) {
    // Track the previous channel title to detect actual channel switches
    const prevChannelTitleRef = useRef(null);
    const didAutoPlayRef = useRef(false);
    const userSelectedRef = useRef(false);

    // Expose a way for parent to mark user selection
    React.useEffect(() => {
        userSelectedRef.current = false;
    }, [channel.title]); // reset on channel switch

    // Sync playlist length to state machine and set first qualifying item ONLY on true channel switch or first load
    useEffect(() => {
        try {
            const playlist = channel.playlist || [];
            const length = playlist.length;
            const channelTitle = channel.title;

            if (length > 0) {
                // Update playlist length in state machine
                if (tvStateMachine.getState().playlistLength !== length) {
                    tvStateMachine.transition('SET_PLAYLIST_LENGTH', length);
                }

                // Only auto-play if channel changed or first load, and not after user selection
                if ((channelTitle && prevChannelTitleRef.current !== channelTitle) && !userSelectedRef.current) {
                    const firstQualifying = tvStateMachine.getFirstQualifyingItem(playlist);
                    tvStateMachine.transition('SET_CURRENT_ITEM', firstQualifying);
                    setCurrentItem(firstQualifying);
                    tvStateMachine.transition('PLAY');
                    didAutoPlayRef.current = true;
                    prevChannelTitleRef.current = channelTitle;
                }
            }
        } catch (e) {
            console.error('Error syncing playlist length:', e);
        }
    }, [channel.playlist, channel.title, setCurrentItem]);

  // Sync currentItem from state machine
    useEffect(() => {
        try {
            const unsub = tvStateMachine.subscribe((event, prev, next) => {
                try {
                    if (prev.currentItem !== next.currentItem) {
                        setCurrentItem(next.currentItem);
                    }
                } catch (e) {
                    console.error('Error in channel state sync subscription:', e);
                }
            });
            // Set initial value
            setCurrentItem(tvStateMachine.getState().currentItem);
            return unsub;
        } catch (e) {
            console.error('Error setting up channel state sync:', e);
            return () => {};
        }
    }, [setCurrentItem]);

    // Mark user selection when onItemSelect is called (parent should set this ref)
    return {
        markUserSelected: () => { userSelectedRef.current = true; }
    };

    // No return needed
}

/**
 * Custom hook to manage channel API calls
 * Provides getChannel and getChannels functions with proper error handling
 * @param {string} baseUrl - Base URL for API calls
 * @param {string} currentChannelId - ID of current channel
 * @param {Function} setChannel - Function to update channel state
 * @param {Function} setChannels - Function to update channels list state
 * @returns {Object} Object with getChannel and getChannels functions
 */
export function useChannelApi(baseUrl, currentChannelId, setChannel, setChannels) {
    const getChannel = useCallback(() => {
        if (!currentChannelId) {
            return;
        }
        fetch(baseUrl + '/node/' + currentChannelId + '/channel')
            .then(response => {
                return response.json();
            })
            .then(data => {
                setChannel(data);
            })
            .catch(error => {
                console.error('[useChannelApi] Error fetching channel:', error);
            });
    }, [baseUrl, currentChannelId, setChannel]);

    const getChannels = useCallback(() => {
        fetch(baseUrl + '/tv/channels')
            .then(response => {
                return response.json();
            })
            .then(data => {
                // Ensure data is an array before setting
                if (Array.isArray(data.channels)) {
                    setChannels(data.channels);
                } else {
                    console.error('[useChannelApi] Expected channels to be an array, got:', data);
                    setChannels([]);
                }
            })
            .catch(error => {
                console.error('[useChannelApi] Error fetching channels:', error);
                setChannels([]);
            });
    }, [baseUrl, setChannels]);

    return { getChannel, getChannels };
}

/**
 * Handler factory to set current item and reset progress if needed
 * Resets progress if item is >= 95% complete before selecting
 * @param {Object} channel - Channel object
 * @param {Function} updateProgress - Function to update progress
 * @param {string} baseUrl - Base URL for API calls
 * @param {Function} setCurrentItem - Function to update current item
 * @returns {Function} Handler function (index, itemIndex) => void
 */
export const setCurrentItemAndResetIfNeeded = (channel, updateProgress, baseUrl, setCurrentItem) => (index, itemIndex) => {
    const item = channel.playlist[itemIndex];
    if (item && typeof item.progress === 'number' && item.progress >= 95 && item.id) {
        // Reset progress in localStorage
        try {
            localStorage.setItem('tv_progress_' + item.id, '0');
        } catch (e) {}
        updateProgress(item.id, 0);
    }
    setCurrentItem(index);
};

/**
 * ChannelItems component - Renders playlist items with progress bars
 * @param {Object} props - Component props
 * @param {Object} props.channel - Channel object with playlist array
 * @param {number} props.currentItem - Index of currently playing item
 * @param {Function} props.onItemSelect - Callback when item is clicked
 * @param {Function} props.handleResetProgress - Callback to reset item progress
 * @param {boolean} props.focused - Whether this component is focused
 * @param {Object} props.headingRef - Ref for heading element
 * @param {boolean} props.open - Whether accordion is open
 * @param {Function} props.setOpen - Function to toggle accordion
 * @param {number} props.scrollableHeight - Height for scrollable area
 * @param {Function} props.getFocusState - Function to get focus state for items
 * @param {Object} props.tvRef - Reference to TV player for seeking
 */
function ChannelItems({ channel, currentItem, onItemSelect, handleResetProgress, focused, headingRef, open, setOpen, scrollableHeight, getFocusState, tvRef }) {
    // Try different possible property names for playlist, memoized to avoid changing dependencies on every render
    const playlist = React.useMemo(() => channel?.playlist || channel?.videos || channel?.items || [], [channel?.playlist, channel?.videos, channel?.items]);

    // State to hold per-item progress loaded from localStorage
    const [progressMap, setProgressMap] = React.useState({});

    // On mount or when playlist changes, load progress for each item from localStorage
    React.useEffect(() => {
        if (!Array.isArray(playlist)) return;
        const map = {};
        playlist.forEach(item => {
            if (item && item.id) {
                let prog = 0;
                try {
                    const stored = localStorage.getItem('tv_progress_' + item.id);
                    if (stored && !isNaN(Number(stored))) {
                        prog = Number(stored);
                    }
                } catch (e) {}
                map[item.id] = prog;
            }
        });
        setProgressMap(map);
    }, [playlist]);

    // Poll localStorage to update progress display during playback
    React.useEffect(() => {
        if (!Array.isArray(playlist) || playlist.length === 0) return;
        const interval = setInterval(() => {
            const updates = {};
            let hasChanges = false;
            playlist.forEach(item => {
                if (item && item.id) {
                    try {
                        const stored = localStorage.getItem('tv_progress_' + item.id);
                        if (stored && !isNaN(Number(stored))) {
                            const prog = Number(stored);
                            if (progressMap[item.id] !== prog) {
                                updates[item.id] = prog;
                                hasChanges = true;
                            }
                        }
                    } catch (e) {}
                }
            });
            if (hasChanges) {
                setProgressMap(prev => ({ ...prev, ...updates }));
            }
        }, 1000);
        return () => clearInterval(interval);
    }, [playlist, progressMap]);

    // Helper to get progress percent for an item, using clip boundaries
    const getProgress = (item) => {
        if (!item || !item.id) return 0;
        const percent = progressMap[item.id] || 0;
        // If item has startTs or endTs, calculate progress relative to clip
        // If clip boundaries are valid, percent is already relative to the clip
        return percent;
    };

    // Handler for clicking the progress bar
    const handleProgressBarClick = (e, item, index) => {
        if (index !== currentItem) {
            return; // Only allow seeking on current item
        }
        const bar = e.currentTarget;
        const rect = bar.getBoundingClientRect();
        const x = e.clientX - rect.left;
        // Calculate seek time relative to startTs and endTs (clip boundaries)
        let startTs = 0;
        let endTs = null;
        if (item.startTs) {
            const parts = item.startTs.split(':').map(Number);
            if (parts.length >= 3) {
                startTs = parts[0] * 3600 + parts[1] * 60 + parts[2];
            }
        }
        if (item.endTs) {
            const parts = item.endTs.split(':').map(Number);
            if (parts.length >= 3) {
                endTs = parts[0] * 3600 + parts[1] * 60 + parts[2];
            }
        }
        const duration = typeof item.duration === 'number' ? item.duration : parseInt(item.duration, 10);
        const clipEnd = endTs !== null ? endTs : duration;
        const clipLength = clipEnd - startTs;
        // Use integer math for accuracy: scale up to 10000, then divide
        let percent = Math.floor((x * 10000) / rect.width);
        percent = Math.max(0, Math.min(10000, percent));
        let seekSeconds = startTs;
        if (clipLength > 0) {
            seekSeconds = startTs + Math.floor((percent * clipLength) / 10000);
        }
        // Clamp seekSeconds to just before clipEnd to avoid floating point rounding up
        if (seekSeconds > clipEnd - 1) seekSeconds = clipEnd - 1;
        if (seekSeconds < startTs) seekSeconds = startTs;
        // Recalculate percent based on clamped seekSeconds, using integer math
        let clampedPercent = 0;
        if (clipLength > 0) {
            clampedPercent = Math.floor(((seekSeconds - startTs) * 100) / clipLength);
            clampedPercent = Math.max(0, Math.min(100, clampedPercent));
        }
        try {
            localStorage.setItem('tv_progress_' + item.id, String(clampedPercent));
            setProgressMap(prev => ({ ...prev, [item.id]: clampedPercent })); // force re-render
        } catch (e) {
            console.warn('[ProgressBarClick] localStorage error', e);
        }
        if (typeof channel.updateProgress === 'function') {
            channel.updateProgress(item.id, clampedPercent);
        }
        // Set desiredStartTimeRef for YouTube.js to trigger reload/seek
        if (tvRef && tvRef.current) {
            // Try direct seek on YouTube player first
            if (tvRef.current.players?.youtube?.seekTo) {
                try {
                    tvRef.current.players.youtube.seekTo(seekSeconds, true);
                } catch (e) {
                    console.error('[ChannelItems] Error seeking directly:', e);
                }
            }
            // Also set desiredStartTimeRef as backup
            if (tvRef.current.desiredStartTimeRef) {
                tvRef.current.desiredStartTimeRef.current = seekSeconds;
                if (typeof tvRef.current.triggerSeekFromProgressBar === 'function') {
                    tvRef.current.triggerSeekFromProgressBar();
                }
            }
        } else {
            console.warn('[ChannelItems] tvRef not available');
        }
    };

    return (
        <AccordionMenu heading="Channel" className="channel-menu" focused={focused} headingRef={headingRef} open={open} setOpen={setOpen} scrollableHeight={scrollableHeight}>
            <ul>
                {(() => {
                    const noItems = !channel || !Array.isArray(playlist) || playlist.length === 0;
                    return noItems;
                })() && (
                    <li className="channel-items-empty">
                        No items in this channel
                    </li>
                )}
                {playlist.map((item, index) => (
                    <li key={item?.id || index} className={`${index === currentItem ? 'current' : ''} ${getFocusState ? (getFocusState('channel', index, false) ? 'focused' : '') : ''}`}>
                        <button
                            onClick={e => {
                                e.preventDefault();
                                if (typeof onItemSelect === 'function') {
                                    onItemSelect(index);
                                }
                            }}
                            className={`channel-item-button${index === currentItem ? ' current' : ''}`}
                        >
                            {item?.posterUrl && (
                                <img src={item.posterUrl} alt={item?.name || 'Video'} />
                            )}
                            {item?.name || 'Untitled'}
                        </button>
                        <p className="channel-item-duration">Duration: {formatDuration(typeof item?.duration === 'number' ? item.duration : (typeof item?.duration === 'string' ? parseInt(item.duration, 10) : 0))}</p>
                        <div className="progress-row">
                            <button
                                className="progress-refresh-btn"
                                title="Reset Progress"
                                aria-label="Reset Progress"
                                onClick={e => {
                                    e.preventDefault();
                                    if (typeof handleResetProgress === 'function') {
                                        handleResetProgress(item, index, setProgressMap);
                                    }
                                }}
                            >
                                &#8635;
                            </button>
                            <div className="progress-container" style={{flex:1, cursor:'pointer'}} onClick={e => handleProgressBarClick(e, item, index)}>
                                <div className="progress-bar" style={{width: `${getProgress(item)}%`}}></div>
                            </div>
                        </div>
                    </li>
                ))}
            </ul>
        </AccordionMenu>
    );
}

export default ChannelItems;
