/**
 * ChannelList - Channel selection UI component
 * 
 * @component
 * @scope UI for displaying and selecting channels from available channel list
 * @responsibilities
 * - Render list of available channels
 * - Highlight currently selected channel
 * - Handle channel selection clicks
 * - Export hooks for channel initialization and fetching
 * @limits
 * - ❌ Must NOT fetch channel playlist data (TvContainer handles this)
 * - ❌ Must NOT manage tvStateMachine directly
 * - ✅ Should ONLY handle channel list UI and selection
 * @see .cursorrules for full architecture boundaries
 */
import { useEffect } from 'react';
import AccordionMenu from './AccordionMenu';
import './Channels.css';
import { getTvCookieName, getCookie } from '../helpers';

/**
 * Custom hook to fetch channels and set initial channel from cookie
 * @param {Function} getChannels - Function to fetch channels list
 * @param {Array} channels - Current channels array
 * @param {string} currentChannelId - Current channel ID
 * @param {Function} setCurrentChannelId - Function to set channel ID
 */
export function useChannelsInit(getChannels, channels, currentChannelId, setCurrentChannelId) {
    // On mount, fetch channels
    useEffect(() => {
        try {
            getChannels();
        } catch (e) {
            console.error('[useChannelsInit] Error fetching channels:', e);
        }
    }, [getChannels]);

    // When channels are loaded, set initial channel from cookie or default to first
    useEffect(() => {
        try {
            if (channels.length > 0 && !currentChannelId) {
                const cookieName = getTvCookieName();
                const savedChannelId = getCookie(cookieName);
                
                // Check if the saved channel exists in the available channels
                const savedChannelExists = savedChannelId && 
                    channels.some(ch => String(ch.id) === String(savedChannelId));
                
                if (savedChannelExists) {
                    setCurrentChannelId(savedChannelId);
                } else {
                    setCurrentChannelId(channels[0].id);
                }
            }
        } catch (e) {
            console.error('[useChannelsInit] Error setting initial channel:', e);
        }
    }, [channels, currentChannelId, setCurrentChannelId]);
}

/**
 * Handler factory to select a channel and persist to cookie
 * @param {Function} setCurrentChannelId - Function to set current channel ID
 * @returns {Function} Handler function (id) => void
 */
export const onChannelSelect = (setCurrentChannelId) => (id) => {
    setCurrentChannelId(id);
};

/**
 * ChannelList component - Renders channel selection list
 * @param {Object} props - Component props
 * @param {Array} props.channels - Array of channel objects with id and title
 * @param {string} props.currentChannelId - ID of currently selected channel
 * @param {Function} props.onChannelSelect - Callback when channel is selected
 * @param {boolean} props.focused - Whether this accordion section is focused
 * @param {Object} props.headingRef - Ref for heading element (from Accordion)
 * @param {boolean} props.open - Whether accordion is open
 * @param {Function} props.setOpen - Function to toggle accordion
 * @param {number} props.scrollableHeight - Height for scrollable area
 * @param {Function} props.getFocusState - Function to get focus state for items
 */
function ChannelList({ channels, currentChannelId, onChannelSelect, focused, headingRef, open, setOpen, scrollableHeight, getFocusState }) {
    const safeChannels = Array.isArray(channels) ? channels : [];
    return (
        <AccordionMenu heading="Channels" className="channels-menu" focused={focused} headingRef={headingRef} open={open} setOpen={setOpen} scrollableHeight={scrollableHeight}>
            <ul>
                {safeChannels.length === 0 && (
                    <li style={{color:'#888'}}>No channels available</li>
                )}
                {safeChannels.map((item, index) => {
                    const isCurrentChannel = String(item?.id) === String(currentChannelId);
                    return (
                        <li key={item?.id || index} className={`${isCurrentChannel ? 'current' : ''} ${getFocusState ? (getFocusState('channels', index, false) ? 'focused' : '') : ''}`}>
                            <a
                                href="/"
                                onClick={e => {
                                    e.preventDefault();
                                    onChannelSelect(item?.id);
                                }}
                                style={{ color: isCurrentChannel ? '#fff' : '#ccc', fontWeight: isCurrentChannel ? 600 : 400 }}
                            >
                                {item?.title || 'Untitled Channel'}
                            </a>
                        </li>
                    );
                })}
            </ul>
        </AccordionMenu>
    );
}

export default ChannelList;
