/**
 * helpers - Pure utility functions for TV app
 * 
 * @module
 * @scope Reusable pure functions - no side effects, no state management
 * @responsibilities
 * - Parse timestamp strings to seconds (getInSeconds)
 * - Format seconds as HH:MM:SS (formatDuration)
 * - Parse ISO 8601 durations (parseDuration)
 * - Cookie utilities (getTvCookieName, getCookie, setCookie)
 * @limits
 * - ❌ Must NOT contain side effects (except setCookie which modifies document.cookie)
 * - ❌ Must NOT manipulate DOM
 * - ❌ Must NOT manage state
 * - ✅ All exports must be stateless and predictable
 * @see .cursorrules for full architecture boundaries
 */

/**
 * Format seconds as HH:MM:SS or MM:SS
 * @param {number} seconds - Seconds to format
 * @returns {string} Formatted time string
 */
function formatDuration(seconds) {
    seconds = Math.max(0, Math.floor(seconds));
    const h = Math.floor(seconds / 3600);
    const m = Math.floor((seconds % 3600) / 60);
    const s = seconds % 60;
    if (h > 0) {
        return `${h}:${m.toString().padStart(2, '0')}:${s.toString().padStart(2, '0')}`;
    } else {
        return `${m}:${s.toString().padStart(2, '0')}`;
    }
}

/**
 * Parse timestamp string to seconds
 * Supports HH:MM:SS:FF (with frames), HH:MM:SS, and MM:SS formats
 * @param {string} timestamp - Timestamp string to parse
 * @returns {number} Total seconds (including fractional seconds for frames)
 */
function getInSeconds(timestamp) {
    if (!timestamp || typeof timestamp !== 'string') return 0;
    let a = timestamp.split(':');
    // Handle HH:MM:SS:FF format (with frames)
    if (a.length === 4) {
        const framesPerSecond = 30; // Assuming 30fps
        return (+a[0]) * 60 * 60 + (+a[1]) * 60 + (+a[2]) + (+a[3]) / framesPerSecond;
    }
    // Handle HH:MM:SS format
    if (a.length === 3) {
        return (+a[0]) * 60 * 60 + (+a[1]) * 60 + (+a[2]);
    }
    // Handle MM:SS format
    if (a.length === 2) {
        return (+a[0]) * 60 + (+a[1]);
    }
    return 0;
}

/**
 * Parse ISO 8601 duration string (PT1H2M3S) to seconds
 * @param {string} duration - ISO 8601 duration string
 * @returns {number} Total seconds
 */
function parseDuration(duration) {
    if (!duration || typeof duration !== 'string') return 0;
    const match = duration.match(/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/);
    if (!match) return 0;
    const hours = parseInt(match[1] || 0);
    const minutes = parseInt(match[2] || 0);
    const seconds = parseInt(match[3] || 0);
    return hours * 3600 + minutes * 60 + seconds;
}

/**
 * Generate a unique cookie name based on the current page path
 * @param {string} prefix - Prefix for cookie name (default: 'tv_channel')
 * @returns {string} Cookie name with sanitized path
 */
function getTvCookieName(prefix = 'tv_channel') {
    if (typeof window === 'undefined') return prefix;
    const path = window.location.pathname
        .replace(/[^a-zA-Z0-9]/g, '_')  // Replace non-alphanumeric chars with underscore
        .replace(/_+/g, '_')             // Collapse multiple underscores
        .replace(/^_|_$/g, '');          // Trim leading/trailing underscores
    return `${prefix}_${path || 'home'}`;
}

/**
 * Get a cookie value by name
 * @param {string} name - Cookie name
 * @returns {string|null} Cookie value or null if not found
 */
function getCookie(name) {
    if (typeof document === 'undefined') return null;
    const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
    return match ? decodeURIComponent(match[2]) : null;
}

/**
 * Set a cookie with name and value
 * @param {string} name - Cookie name
 * @param {string} value - Cookie value
 * @param {number} days - Expiration in days (default: 365)
 */
function setCookie(name, value, days = 30) {
    if (typeof document === 'undefined') return;
    const expires = new Date(Date.now() + days * 864e5).toUTCString();
    document.cookie = `${name}=${encodeURIComponent(value)}; expires=${expires}; path=/; SameSite=Lax`;
}

export { formatDuration, getInSeconds, parseDuration, getTvCookieName, getCookie, setCookie };
