<?php

namespace Drupal\tv\Entity\Collection;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\Entity\Node;
use Drupal\tv\Entity\TvChannel;
use Drupal\tv\Service\TvChannelServiceInterface;

class TvChannelCollection extends EntityCollection {

    private EntityTypeManagerInterface $entityTypeManager;

    private TvChannelServiceInterface $tvChannelService;

    public function __construct(EntityTypeManagerInterface $entityTypeManager, TvChannelServiceInterface $tvChannelService)
    {
        $this->entityTypeManager = $entityTypeManager;
        $this->tvChannelService = $tvChannelService;
    }

    public function load(): static
    {
        $result = $this->entityTypeManager->getStorage('node')
            ->getQuery()
            ->condition('type', 'tv_channel')
            ->condition('status', 1)
            ->sort('field_weight')
            ->sort('title')
            ->accessCheck()
            ->execute();
        foreach ($result as $nid) {
            $items = $this->tvChannelService->getItems(Node::load($nid));
            if (count($items)) {
              $this->append($nid);
            }
        }
        return $this;
    }

    public function current(): TvChannel
    {
        return (new TvChannel($this->entityTypeManager))
            ->load($this->getIterator()->current());
    }

    /**
     * Sort the collection by weight, then alphabetically by title.
     */
    public function sort(): static
    {
        if (empty($this->data)) {
            return $this;
        }

        // Load all nodes to get their weight and title for sorting
        $nodes = Node::loadMultiple($this->data);

        usort($this->data, function ($a, $b) use ($nodes) {
            $nodeA = $nodes[$a] ?? null;
            $nodeB = $nodes[$b] ?? null;

            if (!$nodeA || !$nodeB) {
                return 0;
            }

            // Get weights (default to 0 if not set)
            $weightA = $nodeA->hasField('field_weight') && !$nodeA->get('field_weight')->isEmpty()
                ? (int) $nodeA->get('field_weight')->value
                : 0;
            $weightB = $nodeB->hasField('field_weight') && !$nodeB->get('field_weight')->isEmpty()
                ? (int) $nodeB->get('field_weight')->value
                : 0;

            // Sort by weight first (ascending)
            if ($weightA !== $weightB) {
                return $weightA <=> $weightB;
            }

            // Then sort alphabetically by title
            return strcasecmp($nodeA->getTitle(), $nodeB->getTitle());
        });

        return $this;
    }
}
