/**
 * LowerThirdOverlay - Display overlay for lower third graphics (news-style captions)
 * 
 * @component
 * @scope Pure presentation component for displaying lower third graphics during video playback
 * @responsibilities
 * - Render header, text, and QR code in lower third position
 * - Show/hide based on visible prop with CSS transition
 * - Position at bottom-right of video player
 * @limits
 * - ❌ Must NOT fetch data or manage timing (TvContainer handles this)
 * - ❌ Must NOT interact with player or state machine
 * - ✅ Should be pure presentation component
 * @see .cursorrules for full architecture boundaries
 */
import React from 'react';
import './LowerThirdOverlay.css';

/**
 * LowerThirdOverlay component - News-style lower third graphics
 * @param {Object} props - Component props
 * @param {string} props.header - Main heading text
 * @param {string} props.text - Subheading or body text
 * @param {string} props.qr - QR code image URL
 * @param {boolean} props.visible - Whether overlay should be visible
 */
function LowerThirdOverlay({ header, text, qr, visible }) {
  return (
    <div
      className={`tv-third-overlay${visible ? ' visible' : ''}`}
      style={{ position: 'absolute', right: 0, bottom: '5%', zIndex: 1000 }}
    >
      <div className="tv-third-overlay-row">
        {qr && <img src={qr} alt="" width={100} height={100} />}
        <div className="tv-third-overlay-content">
          {header && <h1>{header}</h1>}
          {text && <em>{text}</em>}
        </div>
      </div>
    </div>
  );
}

export default LowerThirdOverlay;
