/**
 * YouTube - Provider wrapper for YouTube iframe API
 * 
 * @component
 * @scope Simple wrapper for YouTube API - exposes standardized playback methods
 * @responsibilities
 * - Load and configure YouTube iframe API
 * - Load videos with startSeconds parameter for smooth resuming
 * - Expose methods: play(), pause(), seekTo(), getCurrentTime(), mute(), unmute()
 * - Emit state change events to parent and tvStateMachine
 * - Track user mute preference across video changes
 * @limits
 * - ❌ Must NOT calculate resume positions (Player.js handles this via startSeconds prop)
 * - ❌ Must NOT manage clip boundaries (Player.js handles startTs/endTs)
 * - ❌ Must NOT write progress to localStorage (Player.js handles this)
 * - ❌ Must NOT contain business logic beyond YouTube API interaction
 * @see .cursorrules for full architecture boundaries and future Vimeo.js/Rumble.js patterns
 */

import {getInSeconds, parseDuration} from "../../../../helpers";
import tvStateMachine from '../../../../tvStateMachine';
import React, { useEffect, useRef, useCallback, useState, useLayoutEffect } from 'react';

/**
 * Utility: Find first incomplete video index (progress < 95%)
 * @param {Array} playlist - Array of playlist items
 * @returns {number} Index of first incomplete item, or 0 if all complete
 */
function findFirstIncompleteIndex(playlist) {
    if (!Array.isArray(playlist)) return 0;
    for (let i = 0; i < playlist.length; i++) {
        const item = playlist[i];
        let progress = 0;
        try {
            const stored = localStorage.getItem('tv_progress_' + item.id);
            if (stored && !isNaN(Number(stored))) {
                progress = Number(stored);
            }
        } catch (e) {}
        if (progress < 95) return i;
    }
    return 0;
}

/**
 * YouTube component - Wraps YouTube iframe API for video playback
 * @param {Object} params - Component params
 * @param {Object} params.item - Current item (object or index)
 * @param {Object} params.channel - Channel object with playlist
 * @param {Object} params.tvRef - Reference to expose player methods
 * @param {number} params.startSeconds - Calculated start position from Player.js (includes resume)
 * @param {Function} params.onVideoEnd - Callback when video ends
 * @param {Function} params.onPlayerStateChange - Callback for state changes
 * @param {Function} params.updateProgress - Callback to update progress
 * @param {Function} params.onError - Callback for errors
 */
function YouTube(params) {
        // Allow parent to subscribe to YouTube player state changes
        const onPlayerStateChange = params.onPlayerStateChange;
    // Forever-loop detection: track recent loads
    const loadTimestampsRef = useRef([]);
    const MAX_LOADS = 10;
    const WINDOW_MS = 5000;
    // Track YouTube API readiness
    const [youtubeApiReady, setYoutubeApiReady] = useState(!!window.youtubeApiReady);
    // On initial mount, if params.item is not set or 0, find first incomplete
    let initialItemIndex = params.item;
    if (typeof initialItemIndex !== 'number' || initialItemIndex === 0) {
        initialItemIndex = findFirstIncompleteIndex(params.channel.playlist);
    }
    // Memoize playlist and item for stable hook dependencies
    const playlist = React.useMemo(() => params.channel.playlist || [], [params.channel.playlist]);
    const item = React.useMemo(() => playlist[initialItemIndex] || { id: '', url: '', duration: 0 }, [playlist, initialItemIndex]);

    const updateProgress = params.updateProgress;
    const onVideoEnd = params.onVideoEnd;
    const currentMediaIdRef = useRef(item.id);
    const channelRef = useRef(params.channel);
    // Always call useRef, then use prop if provided
    const localTvRef = useRef(null);
    const tvRef = params.tvRef || localTvRef;

    // Track if user has explicitly unmuted (persists across video changes)
    const userHasUnmutedRef = useRef(false);

    // Monitor state machine for unmute action
    useEffect(() => {
        const unsubscribe = tvStateMachine.subscribe((event, prev, next) => {
            if (event === 'UNMUTE') {
                userHasUnmutedRef.current = true;
            } else if (event === 'MUTE') {
                userHasUnmutedRef.current = false;
            }
        });
        return unsubscribe;
    }, []);

    // Expose getCurrentTime for parent (TvContainer)
    React.useEffect(() => {
        if (tvRef && tvRef.current) {
            tvRef.current.getCurrentTime = () => {
                if (tvRef.current.defaultPlayer && typeof tvRef.current.defaultPlayer.getCurrentTime === 'function') {
                    return tvRef.current.defaultPlayer.getCurrentTime();
                }
                if (tvRef.current.players && tvRef.current.players.youtube && typeof tvRef.current.players.youtube.getCurrentTime === 'function') {
                    return tvRef.current.players.youtube.getCurrentTime();
                }
                return 0;
            };
        }
        // Also attach to the player instance directly if available
        if (tvRef && tvRef.current && tvRef.current.defaultPlayer) {
            tvRef.current.defaultPlayer.getCurrentTimeFromRef = tvRef.current.getCurrentTime;
        }
    }, [tvRef]);
    const desiredStartTimeRef = useRef(null);
    // Flag to block progress updates until initial seek is done
    const isInitialSeek = useRef(false);
    // Store the desired start time for initial seek
    const initialSeekTime = useRef(0);
    // Track if we are waiting for seekTo to complete
    const waitingForSeek = useRef(false);
    // Prevent rapid reloads: only load if videoId or item.id is different
    const lastLoadedIdRef = useRef(null);

    // Load the selected video when params.item changes (keyboard/mouse navigation)
    useEffect(() => {
        const tv = tvRef.current;
        const playlist = channelRef.current.playlist || [];
        // Handle both cases: params.item as object or as index
        let vid;
        if (typeof params.item === 'object' && params.item !== null) {
            vid = params.item;
        } else {
            const idx = typeof params.item === 'number' ? params.item : initialItemIndex;
            vid = playlist[idx];
        }
        if (!tv || !vid) return;
        const videoId = getVideoId(vid.url);
        const duration = typeof vid.duration === 'number' ? vid.duration : parseDuration(vid.duration);
        // Use startSeconds from params (already calculated with resume position in Player.js)
        let startSeconds = params.startSeconds || 0;
        let clipStartSeconds = vid.startTs ? getInSeconds(vid.startTs) : 0;
        let clipEndSeconds = vid.endTs ? getInSeconds(vid.endTs) : duration;
        // Clamp startSeconds to clip boundaries
        if (startSeconds < clipStartSeconds) startSeconds = clipStartSeconds;
        if (startSeconds > clipEndSeconds) startSeconds = clipEndSeconds;
        // Only reload if the videoId or item.id is different from last loaded
        if (lastLoadedIdRef.current === vid.id) return;
        lastLoadedIdRef.current = vid.id;
        // --- Forever-loop detection logic ---
        const now = Date.now();
        loadTimestampsRef.current.push(now);
        // Remove timestamps older than WINDOW_MS
        loadTimestampsRef.current = loadTimestampsRef.current.filter(ts => now - ts < WINDOW_MS);
        if (loadTimestampsRef.current.length > MAX_LOADS) {
            // Too many loads in a short window: throw error
            if (typeof params.onError === 'function') {
                params.onError(new Error('TV Player detected a forever-loop: too many new items loaded in rapid succession.'));
            } else {
                // eslint-disable-next-line no-console
                console.error('TV Player detected a forever-loop: too many new items loaded in rapid succession.');
            }
            return;
        }
        // Block progress updates until initial seek is done
        isInitialSeek.current = true;
        initialSeekTime.current = startSeconds;
        waitingForSeek.current = false;

        // Retry logic for slow YouTube API/player loads
        let retries = 0;
        const maxRetries = 10;
        function tryLoadVideo() {
            if (!tv.players || !tv.players.youtube || typeof tv.players.youtube.loadVideoById !== 'function') {
                if (retries < maxRetries) {
                    retries++;
                    setTimeout(tryLoadVideo, 200);
                } else {
                    // ...removed debug log...
                }
                return;
            }
            // Load video with start time parameter for smoother experience
            tv.players.youtube.loadVideoById({
                videoId: videoId,
                startSeconds: startSeconds
            });
            // Auto-play if the state machine indicates playing
            setTimeout(() => {
                // Only auto-mute if user hasn't explicitly unmuted
                if (tv.players.youtube && typeof tv.players.youtube.mute === 'function' && !userHasUnmutedRef.current) {
                    tv.players.youtube.mute(); // Auto-mute for autoplay unless user has unmuted
                }
                if (tvStateMachine.getState().playerState === 'playing' && tv.players.youtube && typeof tv.players.youtube.playVideo === 'function') {
                    tv.players.youtube.playVideo();
                }
            }, 3000);
        }
        tryLoadVideo();
    }, [params, initialItemIndex, tvRef, updateProgress]);

    // Expose ref for parent (ChannelItems) to trigger seek
    if (params.tvRef && params.tvRef.current) {
        params.tvRef.current.desiredStartTimeRef = desiredStartTimeRef;
        params.tvRef.current.triggerSeekFromProgressBar = () => {
            if (desiredStartTimeRef.current != null) {
                reloadForProgressBarSeek();
            }
        };
    }

    // Polling interval to robustly sync player state with tvStateMachine
    useEffect(() => {
        let pollInterval = null;
        let retryCount = 0;
        function startPolling() {
            const tv = tvRef.current;
            if (!tv || !tv.players || !tv.players.youtube) {
                if (retryCount < 20) {
                    retryCount++;
                    setTimeout(startPolling, 250);
                }
                return;
            }
            let lastState = null;
            pollInterval = setInterval(() => {
                const player = tv.players.youtube;
                if (typeof player.getPlayerState === 'function') {
                    const state = player.getPlayerState();
                    let mappedState = null;
                    switch (state) {
                        case 1: mappedState = 'playing'; break;
                        case 2: mappedState = 'paused'; break;
                        case 0: mappedState = 'ended'; break;
                        case 3: mappedState = 'buffering'; break;
                        case 5: mappedState = 'cued'; break;
                        default: mappedState = 'idle';
                    }
                    if (mappedState !== lastState) {
                        lastState = mappedState;
                        switch (mappedState) {
                            case 'playing':
                                tvStateMachine.transition('PLAY');
                                break;
                            case 'paused':
                                tvStateMachine.transition('PAUSE');
                                break;
                            case 'ended':
                                tvStateMachine.transition('END');
                                break;
                            case 'buffering':
                                tvStateMachine.transition('BUFFERING');
                                break;
                            case 'cued':
                                tvStateMachine.transition('CUED');
                                break;
                            case 'idle':
                                tvStateMachine.transition('IDLE');
                                break;
                            default:
                                break;
                        }
                    }
                }
            }, 500);
        }
        startPolling();
        return () => {
            if (pollInterval) clearInterval(pollInterval);
        };
    }, [tvRef]);

    // Helper to reload/seek to new time from progress bar
    const reloadForProgressBarSeek = useCallback(() => {
        const tv = tvRef.current;
        if (!tv || !tv.players || !tv.players.youtube) {
            return;
        }
        const playlist = channelRef.current.playlist || [];
        const vid = playlist[params.item];
        if (!vid) {
            return;
        }
        const videoId = getVideoId(vid.url);
        let seekSeconds = desiredStartTimeRef.current;
        if (typeof seekSeconds !== 'number' || isNaN(seekSeconds)) {
            return;
        }
        // Block progress updates until initial seek is done
        isInitialSeek.current = true;
        initialSeekTime.current = seekSeconds;
        waitingForSeek.current = false;
        // Pause before reload
        if (typeof tv.players.youtube.pauseVideo === 'function') {
            tv.players.youtube.pauseVideo();
        }
        setTimeout(() => {
            tv.players.youtube.loadVideoById({ videoId, startSeconds: seekSeconds });
            if (typeof tv.players.youtube.playVideo === 'function') tv.players.youtube.playVideo();
            desiredStartTimeRef.current = null;
        }, 100);
    }, [params.item, tvRef]);

    useEffect(() => {
        currentMediaIdRef.current = item ? item.id : undefined;
    }, [item]);

    // Sync quality changes
    useEffect(() => {
        const unsubscribe = tvStateMachine.subscribe((event, prev, next) => {
            if (prev.quality !== next.quality) {
                const tv = tvRef.current;
                if (tv && tv.players && tv.players.youtube) {
                    const player = tv.players.youtube;
                    if (typeof player.getCurrentTime === 'function' && typeof player.loadVideoById === 'function' && typeof player.getVideoData === 'function') {
                        const currentTime = player.getCurrentTime();
                        const videoData = player.getVideoData();
                        if (videoData && videoData.video_id) {
                            const ytQuality = next.quality === 'low' ? 'small' : next.quality === 'med' ? 'large' : 'auto';
                            player.loadVideoById({videoId: videoData.video_id, startSeconds: currentTime, suggestedQuality: ytQuality});
                        }
                    }
                }
            }
        });
        return unsubscribe;
    }, [tvRef]);

    // Per-video progress interval: clear and set only for current video
    // No per-video progress interval logic here; handled by parent/Player.js

    function getVideoId(url) {
        if (!url || typeof url !== 'string') return '';
        const videoId = url.split('v=')[1];
        if (!videoId) return '';
        const ampersandPosition = videoId.indexOf('&');
        if (ampersandPosition !== -1) {
            return videoId.substring(0, ampersandPosition);
        }
        return videoId;
    }

    // Load the YouTube API if not already loaded.
    if (!document.getElementById('iframe-demo')) {
        let tag = document.createElement('script');
        tag.id = 'iframe-demo';
        tag.src = 'https://www.youtube.com/iframe_api';
        let firstScriptTag = document.getElementsByTagName('script')[0];
        firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
    }

    // Configure the YouTube API & TV Player embeds.
    // Notify parent and trigger state machine transitions on YouTube player state changes
    const handleStateChange = useCallback((event, tv) => {
        if (typeof window.YT !== 'undefined') {
            let mappedState = null;
            if (event.data === window.YT.PlayerState.PLAYING) mappedState = 'playing';
            else if (event.data === window.YT.PlayerState.PAUSED) mappedState = 'paused';
            else if (event.data === window.YT.PlayerState.ENDED) mappedState = 'ended';
            else if (event.data === window.YT.PlayerState.BUFFERING) mappedState = 'buffering';
            else if (event.data === window.YT.PlayerState.UNSTARTED) mappedState = 'idle';
            else if (event.data === window.YT.PlayerState.CUED) mappedState = 'cued';
            if (onPlayerStateChange && mappedState) {
                onPlayerStateChange(mappedState);
            }
            // Restore state machine transitions
            switch (event.data) {
                case window.YT.PlayerState.PLAYING:
                    if (window.tvStateMachine) window.tvStateMachine.transition('PLAY');
                    break;
                case window.YT.PlayerState.PAUSED:
                    if (window.tvStateMachine) window.tvStateMachine.transition('PAUSE');
                    break;
                case window.YT.PlayerState.ENDED:
                    if (window.tvStateMachine) window.tvStateMachine.transition('END');
                    if (typeof onVideoEnd === 'function') onVideoEnd();
                    break;
                case window.YT.PlayerState.BUFFERING:
                    if (window.tvStateMachine) window.tvStateMachine.transition('BUFFERING');
                    break;
                case window.YT.PlayerState.UNSTARTED:
                    if (window.tvStateMachine) window.tvStateMachine.transition('IDLE');
                    break;
                case window.YT.PlayerState.CUED:
                    if (window.tvStateMachine) window.tvStateMachine.transition('CUED');
                    if (params.startSeconds > 0) {
                        event.target.seekTo(params.startSeconds, true);
                    }
                    break;
                default:
                    break;
            }
        }
    }, [onPlayerStateChange, onVideoEnd, params.startSeconds]);

    // No loadVideo logic here; parent should control video loading

    useEffect(() => {
        // Load the YouTube API if not already loaded.
        if (!document.getElementById('iframe-demo')) {
            let tag = document.createElement('script');
            tag.id = 'iframe-demo';
            tag.src = 'https://www.youtube.com/iframe_api';
            let firstScriptTag = document.getElementsByTagName('script')[0];
            firstScriptTag.parentNode.insertBefore(tag, firstScriptTag);
        }
    }, []);

    // eslint-disable-next-line react-hooks/exhaustive-deps

    useLayoutEffect(() => {
        function initializePlayer() {
            const tv = tvRef.current;
            if (!tv) {
                console.warn('[YouTube] tvRef.current is null, cannot initialize player');
                return;
            }
            if (!tv.tagName || tv.tagName.toLowerCase() !== 'div') {
                console.warn('[YouTube] tvRef.current is not a valid DOM element, cannot initialize player', tv);
                return;
            }
            if (!tv.isConnected) {
                console.warn('[YouTube] tvRef.current is not connected to the document, cannot initialize player');
                return;
            }
            // Only create if not already exists and YT is available
            if (!tv.players && typeof window.YT !== 'undefined') {
                try {
                    tv.players = tv.players || {};
                    tv.players.youtube = new window.YT.Player(tv, {
                        height: '100%',
                        width: '100%',
                        playerVars: {
                            rel: 0,
                            modestbranding: 1,
                            fs: 0,
                            disablekb: 1,
                            autoplay: 1,
                            mute: 1, // Force mute for autoplay
                            enablejsapi: 1,
                            controls: 0,
                        },
                        events: {
                            onReady: function(event) {
                                tv.players.youtube = event.target;
                                if (tvRef && tvRef.current) {
                                    tvRef.current.players = tv.players;
                                    tvRef.current.defaultPlayer = event.target;
                                    tvRef.current.getCurrentTime = () => {
                                        if (tvRef.current.defaultPlayer && typeof tvRef.current.defaultPlayer.getCurrentTime === 'function') {
                                            return tvRef.current.defaultPlayer.getCurrentTime();
                                        }
                                        if (tvRef.current.players && tvRef.current.players.youtube && typeof tvRef.current.players.youtube.getCurrentTime === 'function') {
                                            return tvRef.current.players.youtube.getCurrentTime();
                                        }
                                        return 0;
                                    };
                                    tvRef.current.defaultPlayer.getCurrentTimeFromRef = tvRef.current.getCurrentTime;
                                }
                                // Auto-mute on first load to ensure autoplay (unless user has unmuted)
                                if (typeof tv.players.youtube.mute === 'function' && !userHasUnmutedRef.current) {
                                    tv.players.youtube.mute();
                                }
                                // Auto-load the video for the initial item if present
                                const videoId = getVideoId(item.url);
                                if (videoId) {
                                    // Load will happen in useEffect
                                } else {
                                    console.warn('[YouTube] No valid videoId to auto-load on player ready for item:', item);
                                }
                            },
                            onStateChange: (event) => {
                                handleStateChange(event, tv);
                            }
                        }
                    });
                    tv.players.youtube.play = function () {
                        tv.players.youtube.playVideo();
                    };
                    tv.players.youtube.pause = function () {
                        tv.players.youtube.pauseVideo();
                    };
                    // Always set defaultPlayer on the DOM node ref for controls compatibility
                    if (tv && tv.players && tv.players.youtube) {
                        tv.defaultPlayer = tv.players.youtube;
                        if (tvRef && tvRef.current) {
                            tvRef.current.getCurrentTime = () => {
                                if (tvRef.current.defaultPlayer && typeof tvRef.current.defaultPlayer.getCurrentTime === 'function') {
                                    return tvRef.current.defaultPlayer.getCurrentTime();
                                }
                                if (tvRef.current.players && tvRef.current.players.youtube && typeof tvRef.current.players.youtube.getCurrentTime === 'function') {
                                    return tvRef.current.players.youtube.getCurrentTime();
                                }
                                return 0;
                            };
                            tvRef.current.defaultPlayer.getCurrentTimeFromRef = tvRef.current.getCurrentTime;
                        }
                    }
                } catch (e) {
                    console.error('[YouTube] Error initializing YouTube player:', e);
                }
            } else if (tv.players && tv.players.youtube && typeof tv.players.youtube.loadVideoById === 'function') {
                // Player already exists: load new video if item changes
                try {
                    const videoId = getVideoId(item.url);
                    if (videoId) {
                        tv.players.youtube.loadVideoById({ videoId });
                    } else {
                        console.warn('[YouTube] No valid videoId to load for item:', item);
                    }
                } catch (e) {
                    console.error('[YouTube] Error loading new video into existing player:', e);
                }
            } else {
                console.warn('[YouTube] window.YT is not defined, cannot initialize player');
            }
        }
        if (!window.youtubeApiReady) {
            window.onYouTubeIframeAPIReady = function () {
                window.youtubeApiReady = true;
                setYoutubeApiReady(true);
                // Try to initialize player now that API is ready
                if (tvRef.current && tvRef.current.tagName) {
                    initializePlayer();
                }
            };
        } else {
            // API already ready, initialize immediately if not already done
            if (tvRef.current && tvRef.current.tagName) {
                // Small delay to ensure DOM is stable
                setTimeout(() => {
                    if (tvRef.current && tvRef.current.isConnected) {
                        initializePlayer();
                    }
                }, 0);
            }
        }
    }, [handleStateChange, tvRef, item, youtubeApiReady, params.startSeconds]);

    if (!item.id) {
        return <div>No video at this index</div>;
    }
    if (!item.url || !item.url.includes('youtube.com')) {
        return <div>Not a YouTube video</div>;
    }
    // Render the container div for the YouTube player
    return (
        <div ref={tvRef} style={{ position: 'relative', width: '100%', height: '100%' }} />
    );
}

export default YouTube;
