/**
 * TvContainer - Main application container coordinating all TV app components
 *
 * @component
 * @scope Root coordinator - manages all UI components and state synchronization
 * @responsibilities
 * - Initialize and manage channel/channels state
 * - Coordinate Player, ChannelItems, ChannelList, Controls components
 * - Handle fullscreen API integration
 * - Manage lower third overlays and timing
 * - Subscribe to tvStateMachine for UI updates
 * - Handle video end events and auto-advance logic
 * - Manage keyboard navigation and focus states
 * - Track mouse hover and idle states
 * @limits
 * - ❌ Must NOT contain playback logic (delegate to Player.js)
 * - ❌ Must NOT contain provider API calls (delegate to wrappers)
 * - ✅ Should ONLY coordinate between components and state machine
 * @see .cursorrules for full architecture boundaries
 */
import React, { useState, useRef, useEffect, useLayoutEffect, useCallback } from 'react';
import tvStateMachine from './tvStateMachine';
import Keys from './Components/Keys';
import './Tv.css';
import LowerThirdOverlay from './Components/LowerThirdOverlay';
import './Components/LowerThirdOverlay.css';
import Player from "./Components/Player/Player";
import { useChannelsInit } from "./Components/ChannelList";
import { useChannelApi, useChannelStateSync, handleResetProgress as handleResetProgressFn, useChannelFetch } from "./Components/ChannelItems";
import Accordion from "./Components/Accordion";
import ChannelList from './Components/ChannelList';
import ChannelItems from './Components/ChannelItems';
import Controls from './Components/Controls';
import Mouse from './Components/Mouse';
import { updateProgress as updateProgressFn } from './Components/Player/playbackEvents';
import { getTvCookieName, setCookie } from './helpers';

/**
 * TvContainer component - Main TV application
 * @param {Object} params - Component params
 * @param {string} params.baseUrl - Base URL for API calls
 * @param {Object} params.initialState - Initial state with channel/item IDs
 */
function TvContainer(params) {
    // Player state must be declared before any use
    const [playerState, setPlayerState] = useState(tvStateMachine.getState().playerState);

    // Sync play/pause button with state machine after first qualifying item starts playing
    useEffect(() => {
        const unsub = tvStateMachine.subscribe((next, prev) => {
            if (next.playerState !== prev.playerState && next.playerState === 'playing') {
                setPlayerState('playing');
            }
        });
        return () => unsub();
    }, []);

    // Directly sync playerState from YouTube player for first load/auto-play
    const playerStateRef = useRef(playerState);
    useEffect(() => { playerStateRef.current = playerState; }, [playerState]);
    const handlePlayerStateChange = useCallback((ytState) => {
        // Only update if state actually changes
        if (playerStateRef.current !== ytState) {
            setPlayerState(ytState);
        }
    }, []);
    // Channel state (must be declared before any use)
    const [channel, setChannel] = useState({ title: '', playlist: [] });
    // Lower third overlay state
    const [showThirdOverlay, setShowThirdOverlay] = useState(false);
    // Lower third list for the current channel item (cached for playback)
    const [lowerThirdList, setLowerThirdList] = useState([]);
    // Current index in the lower third list
    const [currentThirdIndex, setCurrentThirdIndex] = useState(0);
    
    // Loading overlay state
    const [showLoadingOverlay, setShowLoadingOverlay] = useState(true);
    
    // Remove loading overlay from DOM after animation completes (3s delay + 2s fade = 5s total)
    useEffect(() => {
        const timer = setTimeout(() => {
            setShowLoadingOverlay(false);
        }, 5000);
        return () => clearTimeout(timer);
    }, []);

    // Removed duplicate END/reset logic. All END transitions and progress reset are handled after handleVideoEnd.

    // Sync React fullscreen state with browser fullscreen API (handles Esc key exit)
    React.useEffect(() => {
        function handleFullscreenChange() {
            const isFullscreen = document.fullscreenElement || document.webkitFullscreenElement || document.mozFullScreenElement || document.msFullscreenElement;
            if (!isFullscreen) {
                setFullscreen(false);
                tvStateMachine.transition('FULLSCREEN_EXIT');
            }
        }
        document.addEventListener('fullscreenchange', handleFullscreenChange);
        document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
        document.addEventListener('mozfullscreenchange', handleFullscreenChange);
        document.addEventListener('MSFullscreenChange', handleFullscreenChange);
        return () => {
            document.removeEventListener('fullscreenchange', handleFullscreenChange);
            document.removeEventListener('webkitfullscreenchange', handleFullscreenChange);
            document.removeEventListener('mozfullscreenchange', handleFullscreenChange);
            document.removeEventListener('MSFullscreenChange', handleFullscreenChange);
        };
    }, []);

    const playerRef = useRef(null);
    const prevChannelIdRef = useRef(null);

    // Initialize autoHeight with correct value for narrow viewports
    const getInitialAutoHeight = () => {
        if (typeof window !== 'undefined' && window.innerWidth <= 768) {
            // Calculate: player (16:9 aspect) + controls (60px) + sidebar (300px min) + 10px
            const playerHeight = Math.round(window.innerWidth * (9 / 16));
            const controlsHeight = 60;
            const sidebarHeight = 300;
            return `${playerHeight + controlsHeight + sidebarHeight + 10}px`;
        }
        return 700;
    };

    const [autoHeight, setAutoHeight] = useState(getInitialAutoHeight);

    // Monitor actual DOM height changes with ResizeObserver
    useEffect(() => {
        if (!tvContainerRef.current) return;

        const observer = new ResizeObserver(() => {
            // Observer is active to monitor height changes
            // Currently no action needed on resize
        });

        observer.observe(tvContainerRef.current);

        return () => observer.disconnect();
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [autoHeight]); // tvContainerRef is a ref and doesn't need to be in deps

    const [isHovered, setIsHovered] = useState(false);
    const [isIdle, setIsIdle] = useState(false);
    // (duplicate declaration removed)
    const [channels, setChannels] = useState([]);
    // Try to restore last state from localStorage
    const getInitialState = () => {
        try {
            const saved = JSON.parse(localStorage.getItem('tv_last_state'));
            if (saved && typeof saved === 'object') {
                return {
                    currentItem: saved.itemId || saved.itemIndex || null,
                    currentChannelId: saved.channelId || null,
                };
            }
        } catch (e) {}
        return { currentItem: null, currentChannelId: null };
    };
    const initialState = getInitialState();
    // Store currentItem as an index (not ID) for consistency
    const [currentItem, setCurrentItem] = useState(() => {
        if (typeof initialState.currentItem === 'number') return initialState.currentItem;
        // If initialState.currentItem is an ID, map to index
        if (initialState.currentItem && channel && Array.isArray(channel.playlist)) {
            const idx = channel.playlist.findIndex(
                i => String(i.nid || i.id) === String(initialState.currentItem)
            );
            return idx >= 0 ? idx : 0;
        }
        return 0;
    });
    // Only set currentItem to 0 on true initial load or channel switch
    // (declare prevChannelIdRef only once below)
    const [currentChannelId, setCurrentChannelId] = useState(initialState.currentChannelId);
    useEffect(() => {
        if (!channel || !Array.isArray(channel.playlist) || !channel.playlist.length) return;
        // Only run on channel switch
        if (prevChannelIdRef.current !== currentChannelId) {
            prevChannelIdRef.current = currentChannelId;
            if (typeof currentItem !== 'number' || currentItem < 0 || currentItem >= channel.playlist.length) {
                setCurrentItem(0);
            }
        }
    }, [channel, currentChannelId, currentItem]);
    // (duplicate declaration removed)
        // Persist last played channel/item/position to localStorage
        useEffect(() => {
            // Only persist if we have a valid channel and item
            if (!channel || !channel.playlist || !channel.playlist.length) return;
            const item = channel.playlist[currentItem];
            if (!item) return;
            // Log progress percent being stored
            try {
                localStorage.getItem('tv_progress_' + item.id);
            } catch (e) {}
            const stateToSave = {
                channelId: currentChannelId,
                itemIndex: currentItem,
            };
            try {
                localStorage.setItem('tv_last_state', JSON.stringify(stateToSave));
            } catch (e) {}
        }, [currentChannelId, currentItem, channel]);
    const [openIndex, setOpenIndex] = useState(() => channels.length <= 1 ? 1 : 0); // Channel menu if 1 or fewer channels, Channels menu if multiple
    const [currentFocusIndex, setCurrentFocusIndex] = useState(0);

    // Refs for accordion headings
    const channelsHeadingRef = useRef(null);
    const channelHeadingRef = useRef(null);

    // Helper function to determine if an element should be focused based on nav map
    const getFocusState = (section, index = null, isHeading = false) => {
        // Only show focus highlight in keyboard mode (not mouse mode)
        if (isHovered) return false;

        let navIndex = 0;

        // Channels heading
        if (section === 'channels' && isHeading) return currentFocusIndex === 0;
        navIndex++;

        // Channels items (if open)
        if (openIndex === 0) {
            if (section === 'channels' && !isHeading) {
                return currentFocusIndex === navIndex + index;
            }
            navIndex += channels.length;
        }

        // Channel heading
        if (section === 'channel' && isHeading) return currentFocusIndex === navIndex;
        navIndex++;

        // Channel items (if open)
        if (openIndex === 1) {
            if (section === 'channel' && !isHeading) {
                return currentFocusIndex === navIndex + index;
            }
            navIndex += (channel.playlist || []).length;
        }

        // Controls (last 3 items)
        if (section === 'controls') {
            return currentFocusIndex === navIndex + index;
        }

        return false;
    };

    // Helper function to get the focused control index
    const getFocusedControlIndex = () => {
        if (getFocusState('controls', 0)) return 0;
        if (getFocusState('controls', 1)) return 1;
        if (getFocusState('controls', 2)) return 2;
        if (getFocusState('controls', 3)) return 3;
        return -1;
    };

    const [uiActive] = useState(true);
    const baseUrl = params.baseUrl || '';
    // Set window.baseUrl for progress updates
    if (typeof window !== 'undefined') {
        window.baseUrl = baseUrl;
    }
    const [fullscreen, setFullscreen] = useState(false);

    const { tvWidth, playerHeight, tvContainerRef, isNarrowViewport, sidebarHeight, tvContainerHeight } = useTvResize();
    const { getChannel, getChannels } = useChannelApi(baseUrl, currentChannelId, setChannel, setChannels);
    useChannelsInit(getChannels, channels, currentChannelId, setCurrentChannelId);
    useChannelFetch(currentChannelId, getChannel);
    // Get markUserSelected from useChannelStateSync
    const { markUserSelected } = useChannelStateSync(channel, setCurrentItem, setCurrentFocusIndex, openIndex, currentItem);

    // On initial mount, if we restored a channelId, set it (if not already set by default)
    useEffect(() => {
        if (initialState.currentChannelId && !currentChannelId) {
            setCurrentChannelId(initialState.currentChannelId);
        }
    }, [currentChannelId, initialState.currentChannelId, channel]);

    // On channel load (including app load), if all items are complete, reset all progresses and start from the first item
    // Only reset all progresses and start from first item on true channel switch or initial load
    useEffect(() => {
        if (!channel || !channel.playlist || !channel.playlist.length) return;
        // Only run on channel switch
        if (prevChannelIdRef.current !== currentChannelId) {
            prevChannelIdRef.current = currentChannelId;
            // Check if all items are complete (progress >= 95%)
            let allCompleted = true;
            for (const item of channel.playlist) {
                const duration = parseFloat(item.duration) || 0;
                let progress = 0;
                try {
                    const stored = localStorage.getItem('tv_progress_' + item.id);
                    if (stored && !isNaN(Number(stored))) {
                        progress = Number(stored);
                    }
                } catch (e) {}
                const remaining = duration - (progress / 100 * duration);
                if (!(remaining < 30)) {
                    allCompleted = false;
                    break;
                }
            }
            if (allCompleted) {
                // Reset all progresses and start from first item
                channel.playlist.forEach(item => {
                    if (item && item.id) {
                        try {
                            localStorage.setItem('tv_progress_' + item.id, '0');
                        } catch (e) {}
                        item.progress = 0;
                    }
                });
                setCurrentItem(0);
                tvStateMachine.transition('PLAY');
            } else if (
                typeof initialState.currentItem === 'number' &&
                initialState.currentItem < channel.playlist.length &&
                currentItem !== initialState.currentItem
            ) {
                setCurrentItem(initialState.currentItem);
                tvStateMachine.transition('PLAY');
            }
        }
    }, [channel, currentChannelId, initialState.currentItem, currentItem]);

    // On channel load, update playlist progress from localStorage
    useEffect(() => {
        // No longer update playlist with progress; progress is only in localStorage
        // setChannel(prevChannel => {
        //     if (!prevChannel.playlist || prevChannel.playlist.length === 0) return prevChannel;
        //     const updatedPlaylist = loadProgressForPlaylist(prevChannel.playlist);
        //     return { ...prevChannel, playlist: updatedPlaylist };
        // });
    }, [channel.title, currentItem, initialState.currentItem]);
    // Third string from new channel route
    const currentLowerThird = lowerThirdList.length > 0 ? lowerThirdList[currentThirdIndex] : null;
    const thirdHeader = currentLowerThird ? currentLowerThird.heading || '' : '';
    const thirdText = currentLowerThird ? currentLowerThird.text || '' : '';
    const thirdQr = currentLowerThird ? currentLowerThird.qr || '' : '';

    // Lower third cache: { [channelId_itemNid]: [thirdsData] }
    const lowerThirdCacheRef = useRef({});
    const lowerThirdFetchTimerRef = useRef(null);
    // Fetch lower third list 1s after item begins playing, and cache per item
    useEffect(() => {
        // Cleanup timer on unmount or item change
        return () => {
            if (lowerThirdFetchTimerRef.current) {
                clearTimeout(lowerThirdFetchTimerRef.current);
                lowerThirdFetchTimerRef.current = null;
            }
        };
    }, []);

    // Listen for playerState 'playing' and trigger fetch after 1s if needed
    useEffect(() => {
        if (!currentChannelId || typeof currentItem !== 'number' || playerState !== 'playing') return;
        const playlist = channel && Array.isArray(channel.playlist) ? channel.playlist : [];
        if (!playlist.length || currentItem < 0 || currentItem >= playlist.length) return;
        const item = playlist[currentItem];
        if (!item) return;
        const itemNid = item.nid || item.id;
        if (!itemNid) return;
        const cacheKey = `${currentChannelId}_${itemNid}`;
        // If already cached, use it
        if (lowerThirdCacheRef.current[cacheKey]) {
            setLowerThirdList(lowerThirdCacheRef.current[cacheKey]);
            return;
        }
        // Debounce/delay fetch by 1s
        if (lowerThirdFetchTimerRef.current) {
            clearTimeout(lowerThirdFetchTimerRef.current);
        }
        lowerThirdFetchTimerRef.current = setTimeout(() => {
            const url = `/tv/channel/${currentChannelId}/${itemNid}/thirds`;
            fetch(url)
                .then(res => res.json())
                .then(data => {
                    if (Array.isArray(data)) {
                        lowerThirdCacheRef.current[cacheKey] = data;
                        setLowerThirdList(data);
                    } else {
                        lowerThirdCacheRef.current[cacheKey] = [];
                        setLowerThirdList([]);
                    }
                })
                .catch((e) => {
                    lowerThirdCacheRef.current[cacheKey] = [];
                    setLowerThirdList([]);
                });
        }, 1000);
    }, [currentChannelId, currentItem, playerState, channel, channel.playlist]);

    // Reset third index and force overlay to show if list changes and overlay is not visible
    useEffect(() => {
        setCurrentThirdIndex(0);
        if (lowerThirdList.length > 0 && !showThirdOverlay) {
            setShowThirdOverlay(true);
        }
    }, [lowerThirdList, showThirdOverlay]);

    // Custom item selection handler that manages focus
    const onItemSelect = useCallback((item) => {
        let index = -1;
        if (typeof item === 'number') {
            index = item;
        } else if (item && (item.nid || item.id)) {
            index = channel.playlist ? channel.playlist.findIndex(playlistItem => (playlistItem.nid || playlistItem.id) === (item.nid || item.id)) : -1;
        }
        if (index !== -1) {
            if (typeof markUserSelected === 'function') markUserSelected();
            tvStateMachine.transition('PLAY');
            setCurrentItem(index);
            setCurrentFocusIndex(2 + index);
        }
    }, [channel.playlist, setCurrentItem, setCurrentFocusIndex, markUserSelected]);

    const updateProgress = updateProgressFn(channel, setChannel);
    // Pass setProgressMap from ChannelItems to handleResetProgress
    const handleResetProgress = (item, index, setProgressMap) => {
        return handleResetProgressFn(channel, updateProgress, baseUrl, currentItem, playerRef, setProgressMap)(item, index);
    };
    // Auto-advance to next item when video ends
    const handleVideoEnd = () => {
        // Only update progress for the current item
        const finishedItem = channel.playlist[currentItem];
        if (finishedItem && finishedItem.id) {
            try {
                localStorage.setItem('tv_progress_' + finishedItem.id, '100');
            } catch (e) {}
            finishedItem.progress = 100;
        }
        // Always trigger END with the current playlist
        tvStateMachine.transition('END', { playlist: channel.playlist });
    };
        // State machine subscription: handle END transitions, manage playback/progress, and always update playerState
        React.useEffect(() => {
            const unsub = tvStateMachine.subscribe((event, prev, next) => {
                // Always update playerState for Controls
                if (!prev || prev.playerState !== next.playerState) {
                    setPlayerState(next.playerState);
                }
                if (event === 'END') {
                    if (next.playerState === 'ended' && channel && channel.playlist && channel.playlist.length > 0) {
                        // Check if all items are spent (progress >= 95% or remaining < 30s)
                        let allSpent = true;
                        for (const item of channel.playlist) {
                            const duration = parseFloat(item.duration) || 0;
                            let progress = 0;
                            try {
                                const stored = localStorage.getItem('tv_progress_' + item.id);
                                if (stored && !isNaN(Number(stored))) {
                                    progress = Number(stored);
                                }
                            } catch (e) {}
                            const remaining = duration - (progress / 100 * duration);
                            if (!(remaining < 30)) {
                                allSpent = false;
                                break;
                            }
                        }
                        if (allSpent) {
                            channel.playlist.forEach(item => {
                                if (item && item.id) {
                                    try {
                                        localStorage.setItem('tv_progress_' + item.id, '0');
                                    } catch (e) {}
                                    item.progress = 0;
                                }
                            });
                            setCurrentItem(0);
                            setTimeout(() => {
                                tvStateMachine.transition('PLAY');
                            }, 100);
                        } else {
                            // Always find the first qualifying item (wraparound)
                            let firstQualifyingIdx = -1;
                            for (let i = 0; i < channel.playlist.length; i++) {
                                const item = channel.playlist[i];
                                const duration = parseFloat(item.duration) || 0;
                                let progress = 0;
                                try {
                                    const stored = localStorage.getItem('tv_progress_' + item.id);
                                    if (stored && !isNaN(Number(stored))) {
                                        progress = Number(stored);
                                    }
                                } catch (e) {}
                                const remaining = duration - (progress / 100 * duration);
                                if (progress < 95 && remaining >= 30) {
                                    firstQualifyingIdx = i;
                                    break;
                                }
                            }
                            if (firstQualifyingIdx !== -1 && firstQualifyingIdx !== next.currentItem) {
                                setCurrentItem(firstQualifyingIdx);
                                setTimeout(() => {
                                    tvStateMachine.transition('PLAY');
                                }, 100);
                            }
                        }
                    }
                }
            });
            return () => unsub();
        }, [channel, setCurrentItem]);
    const handleVideoLoad = () => {};

    // Handle forever-loop error from Player
    const handlePlayerError = useCallback((err) => {
        // Throw an error instead of alert
        tvStateMachine.transition('PAUSE');
        throw (err && err.message ? new Error(err.message) : new Error('TV Player error'));
    }, []);

    // Custom channel selection handler that sets the first qualifying item
    const onChannelSelect = useCallback((channelId) => {
        // Only switch if it's a different channel
        if (String(channelId) === String(currentChannelId)) {
            return;
        }
        setCurrentChannelId(channelId);
        // Save channel choice to cookie (unique per page path)
        const cookieName = getTvCookieName();
        setCookie(cookieName, channelId);
        // The first qualifying item will be set when the new channel data loads
        // via useChannelStateSync - don't call PLAY here as it causes race conditions
    }, [currentChannelId, setCurrentChannelId]);

    useFullscreenResize(playerRef, setFullscreen);

    // Dynamically set .Tv height based on content (player + controls + sidebar)
    React.useLayoutEffect(() => {
        // Only use 100vh if fullscreen and not paused
        if (fullscreen && playerState !== 'paused') {
            setAutoHeight('100vh');
        } else if (isNarrowViewport) {
            setAutoHeight(tvContainerHeight);
        } else {
            setAutoHeight(700);
        }
    }, [fullscreen, playerState, isNarrowViewport, tvContainerHeight]);

    // Subscribe to state machine events
    React.useEffect(() => {
        const unsubscribe = tvStateMachine.subscribe((event) => {
            if (event === 'RESET_AND_START') {
                // Reset all progress in localStorage only
                setChannel(prevChannel => {
                    prevChannel.playlist.forEach(item => {
                        if (item && item.id) {
                            try {
                                localStorage.setItem('tv_progress_' + item.id, '0');
                            } catch (e) {}
                        }
                    });
                    return prevChannel;
                });
                // Start playing from first item
                tvStateMachine.transition('PLAY');
            }
        });
        return unsubscribe;
    }, [setChannel]);

    // Check if all items are completed on channel load (only when channel ID changes)
    React.useEffect(() => {
        // Only run this check when we've actually switched to a new channel
        if (!currentChannelId || prevChannelIdRef.current === currentChannelId) {
            return;
        }
        // Update the ref to track this channel
        prevChannelIdRef.current = currentChannelId;

        if (channel && channel.playlist && channel.playlist.length > 0) {
            let allCompleted = true;
            for (const item of channel.playlist) {
                const duration = parseFloat(item.duration) || 0;
                let progress = 0;
                try {
                    const stored = localStorage.getItem('tv_progress_' + item.id);
                    if (stored && !isNaN(Number(stored))) {
                        progress = Number(stored);
                    }
                } catch (e) {}
                const remaining = duration - (progress / 100 * duration);
                if (!(remaining < 30)) {
                    allCompleted = false;
                    break;
                }
            }
            if (allCompleted) {
                setCurrentItem(0);
                tvStateMachine.transition('PLAY');
            }
        }
    }, [currentChannelId, channel, baseUrl, setChannel, setCurrentItem]);

    // Update openIndex based on number of channels
    React.useEffect(() => {
        if (channels.length > 1) {
            // Open Channels menu
            setOpenIndex(0);
        } else if (channels.length === 1) {
            // Open Channel menu
            setOpenIndex(1);
        }
    }, [channels.length]);

    // Focus the main container for keyboard navigation
    React.useEffect(() => {
        const focusContainer = () => {
            if (tvContainerRef.current) {
                tvContainerRef.current.focus();
            }
        };

        // Focus immediately
        focusContainer();

        // Also focus after a short delay to handle fullscreen transitions
        const timeoutId = setTimeout(focusContainer, 100);

        return () => clearTimeout(timeoutId);
    }, [fullscreen, tvContainerRef]);

    // Hide controls/sidebar and expand player if fullscreen & idle
    const hideUi = fullscreen && isIdle;

    // ...existing code...

    // Refs for third overlay timers
    const thirdShowTimerRef = useRef(null);
    const thirdHideTimerRef = useRef(null);
    const thirdCycleTimerRef = useRef(null);

    // Helper to clear third overlay timers
    const clearThirdTimers = useCallback(() => {
        if (thirdShowTimerRef.current) {
            clearTimeout(thirdShowTimerRef.current);
            thirdShowTimerRef.current = null;
        }
        if (thirdHideTimerRef.current) {
            clearTimeout(thirdHideTimerRef.current);
            thirdHideTimerRef.current = null;
        }
        if (thirdCycleTimerRef.current) {
            clearInterval(thirdCycleTimerRef.current);
            thirdCycleTimerRef.current = null;
        }
        setShowThirdOverlay(false);
        setCurrentThirdIndex(0);
    }, []);

    // Subscribe to PLAY/PAUSE events to show third overlay (5s after play, show sequence, then wait for next 5m mark)
    const [thirdShouldRender, setThirdShouldRender] = useState(false);
    const [thirdVisible, setThirdVisible] = useState(false);
    const thirdSequenceTimeoutRef = useRef(null);
    const thirdWaitTimeoutRef = useRef(null);
    // Helper to get current video time in seconds
    const getCurrentVideoTime = () => {
        if (playerRef.current && typeof playerRef.current.getCurrentTime === 'function') {
            return playerRef.current.getCurrentTime();
        }
        return 0;
    };
    // Helper to show overlay for the correct duration
    const showOverlayForCurrent = React.useCallback((index) => {
        // Only show overlay if player is playing
        if (playerState !== 'playing') return Promise.resolve();
        const third = lowerThirdList[index];
        const duration = (third && Number(third.duration)) || 10;
        setThirdShouldRender(true);
        setTimeout(() => setThirdVisible(true), 10); // allow mount before adding visible class
        setShowThirdOverlay(true);
        setCurrentThirdIndex(index);
        if (thirdHideTimerRef.current) {
            clearTimeout(thirdHideTimerRef.current);
        }
        return new Promise((resolve) => {
            thirdHideTimerRef.current = setTimeout(() => {
                setThirdVisible(false);
                setShowThirdOverlay(false);
                // Wait for CSS animation out (e.g. 400ms), then unmount
                setTimeout(() => {
                    setThirdShouldRender(false);
                    resolve();
                }, 400);
            }, duration * 1000);
        });
    }, [lowerThirdList, playerState]);

    // Helper to clear all third timers (including new sequence/wait timers)
    const clearAllThirdTimers = useCallback(() => {
        clearThirdTimers();
        if (thirdSequenceTimeoutRef.current) {
            clearTimeout(thirdSequenceTimeoutRef.current);
            thirdSequenceTimeoutRef.current = null;
        }
        if (thirdWaitTimeoutRef.current) {
            clearTimeout(thirdWaitTimeoutRef.current);
            thirdWaitTimeoutRef.current = null;
        }
    }, [clearThirdTimers]);

    // Helper to start the lower third sequence
    const startLowerThirdSequence = useCallback(() => {
        if (!lowerThirdList.length || playerState !== 'playing') {
            return;
        }
        let index = 0;
        const runSequence = async () => {
            while (index < lowerThirdList.length && playerState === 'playing') {
                await showOverlayForCurrent(index);
                index++;
            }
            // Sequence done: wait until next 5m mark
            let now = getCurrentVideoTime();
            // If now is 0, wait 1s and check again to avoid rapid loop
            if (now === 0) {
                thirdWaitTimeoutRef.current = setTimeout(() => {
                    startLowerThirdSequence();
                }, 1000);
                return;
            }
            const next5min = Math.ceil(now / 300) * 300;
            const waitMs = Math.max((next5min - now) * 1000, 0);
            thirdWaitTimeoutRef.current = setTimeout(() => {
                startLowerThirdSequence();
            }, waitMs);
        };
        runSequence();
    }, [lowerThirdList, showOverlayForCurrent, playerState]);

    React.useEffect(() => {
        if (!lowerThirdList.length) {
            return;
        }
        // Subscribe to state machine events
        const unsubscribe = tvStateMachine.subscribe((event, prev, next) => {
            const startedPlaying = (event === 'PLAY' || event === 'NEXT_ITEM') ||
                (prev && next && prev.playerState !== 'playing' && next.playerState === 'playing');
            const stoppedPlaying = event === 'PAUSE' || event === 'SET_CURRENT_ITEM' ||
                (prev && next && prev.playerState === 'playing' && next.playerState !== 'playing');

            if (startedPlaying) {
                clearAllThirdTimers();
                thirdShowTimerRef.current = setTimeout(() => {
                    startLowerThirdSequence();
                }, 30000); // 30s delay before first lower third
            }

            if (stoppedPlaying) {
                clearAllThirdTimers();
                setThirdVisible(false);
                setShowThirdOverlay(false);
                setTimeout(() => {
                    setThirdShouldRender(false);
                }, 400);
            }
        });

        // If the player is already playing when the list loads, trigger the timer immediately
        if (tvStateMachine.getState().playerState === 'playing') {
            clearAllThirdTimers();
            thirdShowTimerRef.current = setTimeout(() => {
                startLowerThirdSequence();
            }, 30000); // 30s delay before first lower third
        }

        return () => {
            unsubscribe();
            clearAllThirdTimers();
            setThirdShouldRender(false);
            setThirdVisible(false);
        };
    }, [lowerThirdList, clearAllThirdTimers, startLowerThirdSequence]);

    // Handle keyboard events on the main container
    const handleKeyDown = React.useCallback((e) => {
        // Prevent default browser focus behavior for Tab to avoid focusing player
        if (e.key === 'Tab' || (e.key === 'Tab' && e.shiftKey)) {
            e.preventDefault();
        }
    }, []);

    return (
        !channel || typeof channel.title === 'undefined' ? null : (
            <div
                className={`Tv${uiActive ? ' Tv--ui-active' : ' Tv--ui-idle'}${fullscreen ? ' fullscreen' : ''}${hideUi ? ' tv-hide-ui' : ''}`}
                ref={tvContainerRef}
                tabIndex={0}
                data-tv-state={JSON.stringify(tvStateMachine.getState())}
                style={{
                    width: '100%',
                    maxWidth: '100vw',
                    height: autoHeight,
                    display: 'flex',
                    flexDirection: isNarrowViewport ? 'column' : 'row',
                    overflow: 'hidden',
                    transition: 'none'
                }}
                onKeyDown={handleKeyDown}
            >
                {/* Loading overlay with gradient */}
                {showLoadingOverlay && (
                    <div className="tv-loading-overlay">
                        {(channel.brand || channel.title) && (
                            <div className="tv-loading-text">
                                {channel.brand || 'Loading'}
                            </div>
                        )}
                    </div>
                )}
                
                {/* Mouse activity tracker for idle/active UI state */}
                <Mouse
                    tvRef={tvContainerRef}
                    isHovered={isHovered}
                    setIsHovered={setIsHovered}
                    isIdle={isIdle}
                    setIsIdle={setIsIdle}
                />
                <Keys
                    channels={channels}
                    channel={channel}
                    setCurrentChannelId={setCurrentChannelId}
                    setCurrentItem={setCurrentItem}
                    setOpenIndex={setOpenIndex}
                    openIndex={openIndex}
                    currentFocusIndex={currentFocusIndex}
                    setCurrentFocusIndex={setCurrentFocusIndex}
                    tvRef={playerRef}
                    setIsIdle={setIsIdle}
                />
                <div style={{
                    display: 'flex',
                    flex: 1,
                    flexDirection: isNarrowViewport ? 'column' : 'row',
                    minWidth: 0,
                    height: isNarrowViewport ? 'auto' : '100%',
                    width: isNarrowViewport ? '100%' : 'auto'
                }}>
                    <div
                        className="tv-main-col"
                        style={{
                            position: 'relative',
                            width: `${tvWidth}px`,
                            minWidth: `${tvWidth}px`,
                            maxWidth: `${tvWidth}px`,
                            height: '100%',
                            flex: '0 0 auto',
                            ...(hideUi ? { width: '100vw', minWidth: 0, maxWidth: '100vw', zIndex: 10 } : {})
                        }}
                    >
                        <div
                            className="tv-player-container"
                            style={{
                                position: 'relative',
                                height: hideUi ? '100vh' : (isNarrowViewport ? 'auto' : `${playerHeight}px`),
                                minHeight: hideUi ? '100vh' : 'auto',
                                maxHeight: hideUi ? '100vh' : 'none',
                                width: hideUi ? '100vw' : '100%',
                                aspectRatio: isNarrowViewport ? '16/9' : undefined,
                                zIndex: hideUi ? 11 : undefined,
                                overflow: 'hidden',
                            }}
                            tabIndex={-1}
                        >
                            <Player
                                channel={channel}
                                item={channel?.playlist?.[currentItem] || currentItem}
                                onVideoEnd={handleVideoEnd}
                                updateProgress={updateProgress}
                                onVideoLoad={handleVideoLoad}
                                onError={handlePlayerError}
                                ref={playerRef}
                                onPlayerStateChange={handlePlayerStateChange}
                            />
                            <LowerThirdOverlay
                              header={thirdHeader}
                              text={thirdText}
                              qr={thirdQr}
                              visible={!isNarrowViewport && thirdShouldRender && thirdVisible && !!currentLowerThird}
                            />
                        </div>
                        {!fullscreen && !hideUi && (
                            <div className="tv-controls-bar">
                                <Controls tvRef={playerRef} containerRef={tvContainerRef} playerState={playerState} fullscreen={fullscreen} controlsFocusIdx={getFocusedControlIndex()} isNarrowViewport={isNarrowViewport} />
                            </div>
                        )}
                        {fullscreen && !hideUi && (
                            <div className="tv-controls-float">
                                <Controls tvRef={playerRef} containerRef={tvContainerRef} playerState={playerState} size="large" fullscreen={fullscreen} controlsFocusIdx={getFocusedControlIndex()} isNarrowViewport={isNarrowViewport} />
                            </div>
                        )}
                    </div>
                    {/* Sidebar right: controls and navigation */}
                    {!hideUi && (
                        <div
                            className={`tv-sidebar${uiActive ? '' : ' tv-sidebar-hidden'}`}
                            style={{
                                minWidth: isNarrowViewport ? '100%' : 240,
                                maxWidth: isNarrowViewport ? '100%' : 340,
                                width: isNarrowViewport ? '100%' : '22vw',
                                flex: isNarrowViewport ? '1 1 0' : '0 0 22vw',
                                minHeight: isNarrowViewport ? 0 : 'auto',
                                transition: 'opacity 0.7s',
                                opacity: uiActive ? 1 : 0,
                                pointerEvents: uiActive ? 'auto' : 'none',
                                background: '#181818',
                                borderLeft: isNarrowViewport ? 'none' : '1px solid #222',
                                borderTop: isNarrowViewport ? '1px solid #222' : 'none',
                                height: isNarrowViewport ? undefined : sidebarHeight,
                                overflowY: 'auto'
                            }}
                        >
                            <Accordion openIndex={openIndex} setOpenIndex={setOpenIndex}>
                                <ChannelList
                                    channels={channels}
                                    currentChannelId={currentChannelId}
                                    onChannelSelect={onChannelSelect}
                                    focused={getFocusState('channels', null, true)}
                                    headingRef={channelsHeadingRef}
                                    getFocusState={getFocusState}
                                />
                                <ChannelItems
                                    channel={channel}
                                    currentItem={currentItem}
                                    onItemSelect={onItemSelect}
                                    handleResetProgress={handleResetProgress}
                                    focused={getFocusState('channel', null, true)}
                                    headingRef={channelHeadingRef}
                                    getFocusState={getFocusState}
                                    tvRef={playerRef}
                                />
                            </Accordion>
                        </div>
                    )}
                </div>
            </div>
        )
    );
}

// Fullscreen monitoring and sync hook
function useFullscreenResize(playerRef, setFullscreen) {
    useEffect(() => {
        function handleFullscreenChange() {
            const isFs = document.fullscreenElement || document.webkitFullscreenElement || document.msFullscreenElement;
            setFullscreen(!!isFs);
        }
        document.addEventListener('fullscreenchange', handleFullscreenChange);
        document.addEventListener('webkitfullscreenchange', handleFullscreenChange);
        document.addEventListener('msfullscreenchange', handleFullscreenChange);
        // Also subscribe to tvStateMachine fullscreen state
        let unsub = null;
        function syncFullscreenState(next) {
            try {
                if (typeof next.fullscreen === 'boolean') setFullscreen(!!next.fullscreen);
            } catch (e) {
                console.error('Error syncing fullscreen state:', e);
            }
        }
        try {
            unsub = tvStateMachine.subscribe((_, __, next) => syncFullscreenState(next));
        } catch (e) {
            console.error('Error subscribing to fullscreen state:', e);
            unsub = () => {};
        }
        return () => {
            document.removeEventListener('fullscreenchange', handleFullscreenChange);
            document.removeEventListener('webkitfullscreenchange', handleFullscreenChange);
            document.removeEventListener('msfullscreenchange', handleFullscreenChange);
            if (unsub) unsub();
        };
    }, [playerRef, setFullscreen]);
}

function useTvResize() {
    // Initialize with immediate calculation to avoid any growth
    const getInitialNarrowViewport = () => typeof window !== 'undefined' && window.innerWidth <= 768;
    const getInitialContainerHeight = () => {
        if (typeof window !== 'undefined' && window.innerWidth <= 768) {
            const estimatedPlayerHeight = Math.round(window.innerWidth * (9 / 16));
            const estimatedControlsHeight = 60;
            const calculatedSidebarHeight = Math.max(200, window.innerHeight - estimatedPlayerHeight - estimatedControlsHeight);
            return `${estimatedPlayerHeight + estimatedControlsHeight + calculatedSidebarHeight}px`;
        }
        return 'auto';
    };

    const [tvWidth, setTvWidth] = useState(() => {
        if (typeof window !== 'undefined' && window.innerWidth <= 768) {
            return window.innerWidth;
        }
        return 0;
    });
    const [controlsHeight, setControlsHeight] = useState(0);
    const [sidebarHeight, setSidebarHeight] = useState('auto');
    const [isNarrowViewport, setIsNarrowViewport] = useState(getInitialNarrowViewport);
    const [tvContainerHeight, setTvContainerHeight] = useState(getInitialContainerHeight);
    const tvContainerRef = useRef(null);

    // Track previous values to prevent unnecessary updates
    const prevTvContainerHeightRef = useRef(tvContainerHeight);
    const resizeTimeoutRef = useRef(null);

    useLayoutEffect(() => {
        function handleResize() {
            // Debounce: clear any pending resize
            if (resizeTimeoutRef.current) {
                clearTimeout(resizeTimeoutRef.current);
            }

            resizeTimeoutRef.current = setTimeout(() => {
                // Check if viewport is narrow (mobile/tablet)
                const narrowThreshold = 768;
                const isNarrow = window.innerWidth <= narrowThreshold;
                setIsNarrowViewport(isNarrow);

            if (tvContainerRef.current) {
                // Try to find sidebar and controls elements
                const sidebar = tvContainerRef.current.parentElement?.querySelector('.tv-sidebar');
                const controls = tvContainerRef.current.querySelector('.tv-controls-bar');
                // Enforce min sidebar width and controls height
                const minSidebar = 200;
                const minControls = 60;

                // On narrow viewports, use full width for player
                if (isNarrow) {
                    setTvWidth(window.innerWidth);
                    // Let sidebar flex to fill remaining space
                    setSidebarHeight('auto');

                    // Calculate: player (16:9 aspect) + controls (60px) + sidebar (300px min) + 10px
                    const playerHeight = Math.round(window.innerWidth * (9 / 16));
                    const controlsHeight = 60;
                    const sidebarHeight = 300;
                    const newHeight = `${playerHeight + controlsHeight + sidebarHeight + 10}px`;
                    if (prevTvContainerHeightRef.current !== newHeight) {
                        prevTvContainerHeightRef.current = newHeight;
                        setTvContainerHeight(newHeight);
                    }
                } else {
                    const actualSidebar = sidebar ? Math.max(sidebar.offsetWidth, minSidebar) : minSidebar;
                    setTvWidth(tvContainerRef.current.offsetWidth - actualSidebar);
                    setSidebarHeight('100%');
                    setTvContainerHeight('auto');
                }

                const actualControls = controls ? Math.max(controls.offsetHeight, minControls) : minControls;
                setControlsHeight(actualControls);
            }
            }, 0); // Execute immediately but debounce rapid successive calls
        }
        handleResize();
        window.addEventListener('resize', handleResize);
        return () => {
            window.removeEventListener('resize', handleResize);
            if (resizeTimeoutRef.current) {
                clearTimeout(resizeTimeoutRef.current);
            }
        };
    }, [isNarrowViewport]);

    const playerHeight = Math.round(tvWidth / (16 / 9));
    const tvHeight = playerHeight + controlsHeight;

    return { tvWidth, tvHeight, playerHeight, tvContainerRef, isNarrowViewport, sidebarHeight, tvContainerHeight };
}

export default TvContainer;
