/**
 * Unit tests for helpers.js utility functions
 * Tests pure functions with no side effects
 */

import { getInSeconds, formatDuration, parseDuration } from '../helpers';

describe('getInSeconds', () => {
  describe('HH:MM:SS format', () => {
    test('converts 01:30:45 to 5445 seconds', () => {
      expect(getInSeconds('01:30:45')).toBe(5445);
    });

    test('converts 00:00:00 to 0 seconds', () => {
      expect(getInSeconds('00:00:00')).toBe(0);
    });

    test('converts 10:00:00 to 36000 seconds', () => {
      expect(getInSeconds('10:00:00')).toBe(36000);
    });

    test('handles single-digit components', () => {
      expect(getInSeconds('1:2:3')).toBe(3723);
    });
  });

  describe('MM:SS format', () => {
    test('converts 05:30 to 330 seconds', () => {
      expect(getInSeconds('05:30')).toBe(330);
    });

    test('converts 00:00 to 0 seconds', () => {
      expect(getInSeconds('00:00')).toBe(0);
    });

    test('converts 59:59 to 3599 seconds', () => {
      expect(getInSeconds('59:59')).toBe(3599);
    });
  });

  describe('HH:MM:SS:FF format (with frames)', () => {
    test('converts frames at 30fps (default)', () => {
      // 00:00:01:15 = 1 second + 15 frames at 30fps = 1.5 seconds
      expect(getInSeconds('00:00:01:15')).toBe(1.5);
    });

    test('converts 00:00:00:30 to 1 second at 30fps', () => {
      expect(getInSeconds('00:00:00:30')).toBe(1);
    });

    test('converts complex timestamp with frames', () => {
      // 01:30:45:10 = 5445 + 10/30 = 5445.333...
      expect(getInSeconds('01:30:45:10')).toBeCloseTo(5445.333, 2);
    });

    test('handles zero frames', () => {
      expect(getInSeconds('00:00:05:00')).toBe(5);
    });
  });

  describe('edge cases', () => {
    test('returns 0 for empty string', () => {
      expect(getInSeconds('')).toBe(0);
    });

    test('returns 0 for invalid format', () => {
      expect(getInSeconds('invalid')).toBe(0);
    });

    test('returns 0 for null/undefined', () => {
      expect(getInSeconds(null)).toBe(0);
      expect(getInSeconds(undefined)).toBe(0);
    });

    test('handles whitespace', () => {
      expect(getInSeconds(' 01:30:45 ')).toBe(5445);
    });
  });
});

describe('formatDuration', () => {
  describe('HH:MM:SS format (1 hour or more)', () => {
    test('formats 3661 seconds as 1:01:01', () => {
      expect(formatDuration(3661)).toBe('1:01:01');
    });

    test('formats 36000 seconds as 10:00:00', () => {
      expect(formatDuration(36000)).toBe('10:00:00');
    });

    test('formats 3600 seconds as 1:00:00', () => {
      expect(formatDuration(3600)).toBe('1:00:00');
    });
  });

  describe('MM:SS format (less than 1 hour)', () => {
    test('formats 330 seconds as 5:30', () => {
      expect(formatDuration(330)).toBe('5:30');
    });

    test('formats 0 seconds as 0:00', () => {
      expect(formatDuration(0)).toBe('0:00');
    });

    test('formats 59 seconds as 0:59', () => {
      expect(formatDuration(59)).toBe('0:59');
    });

    test('formats 3599 seconds as 59:59', () => {
      expect(formatDuration(3599)).toBe('59:59');
    });
  });

  describe('edge cases', () => {
    test('handles negative numbers', () => {
      expect(formatDuration(-100)).toBe('0:00');
    });

    test('handles decimal seconds (rounds down)', () => {
      expect(formatDuration(90.7)).toBe('1:30');
    });

    test('handles very large numbers', () => {
      expect(formatDuration(359999)).toBe('99:59:59');
    });
  });
});

describe('parseDuration', () => {
  describe('ISO 8601 duration format', () => {
    test('parses PT1H2M3S as 3723 seconds', () => {
      expect(parseDuration('PT1H2M3S')).toBe(3723);
    });

    test('parses PT30M as 1800 seconds', () => {
      expect(parseDuration('PT30M')).toBe(1800);
    });

    test('parses PT45S as 45 seconds', () => {
      expect(parseDuration('PT45S')).toBe(45);
    });

    test('parses PT2H as 7200 seconds', () => {
      expect(parseDuration('PT2H')).toBe(7200);
    });

    test('parses PT1H30M45S as 5445 seconds', () => {
      expect(parseDuration('PT1H30M45S')).toBe(5445);
    });

    test('parses PT0S as 0 seconds', () => {
      expect(parseDuration('PT0S')).toBe(0);
    });
  });

  describe('edge cases', () => {
    test('returns 0 for invalid format', () => {
      expect(parseDuration('invalid')).toBe(0);
    });

    test('returns 0 for empty string', () => {
      expect(parseDuration('')).toBe(0);
    });

    test('returns 0 for null/undefined', () => {
      expect(parseDuration(null)).toBe(0);
      expect(parseDuration(undefined)).toBe(0);
    });

    test('handles duration without PT prefix', () => {
      // Should return 0 as invalid format
      expect(parseDuration('1H2M3S')).toBe(0);
    });
  });

  describe('partial durations', () => {
    test('parses PT1H30M (no seconds)', () => {
      expect(parseDuration('PT1H30M')).toBe(5400);
    });

    test('parses PT45M30S (no hours)', () => {
      expect(parseDuration('PT45M30S')).toBe(2730);
    });
  });
});

describe('helper functions integration', () => {
  test('getInSeconds and formatDuration are inverse operations', () => {
    const timeStr = '01:30:45';
    const seconds = getInSeconds(timeStr);
    const formatted = formatDuration(seconds);
    expect(formatted).toBe('1:30:45');
  });

  test('parseDuration and formatDuration work together', () => {
    const isoDuration = 'PT1H30M45S';
    const seconds = parseDuration(isoDuration);
    const formatted = formatDuration(seconds);
    expect(formatted).toBe('1:30:45');
  });

  test('handles round-trip with MM:SS format', () => {
    const timeStr = '05:30';
    const seconds = getInSeconds(timeStr);
    const formatted = formatDuration(seconds);
    expect(formatted).toBe('5:30');
  });
});
