<?php

namespace Drupal\tv\Channel;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;
use Drupal\tv\Entity\Collection\TvChannelCollection;
use Drupal\tv\Service\TvChannelServiceInterface;

/**
 * Collects all channel context providers and returns the first available channel node.
 */
class ChannelContextCollector {
  /**
   * @var \Drupal\tv\Channel\ChannelContextProviderInterface[]
   */
  protected $providers = [];

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager;

  /**
   * @var \Drupal\tv\Service\TvChannelServiceInterface
   */
  protected \Drupal\tv\Service\TvChannelServiceInterface $channelService;

  /**
   * ChannelContextCollector constructor.
   *
   * @param iterable $providers
   *   An iterable of ChannelContextProviderInterface services.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, TvChannelServiceInterface $channelService, $providers = []) {
    if (!is_iterable($providers)) {
      $providers = [];
    }
    foreach ($providers as $provider) {
      $this->providers[] = $provider;
    }
    $this->entityTypeManager = $entityTypeManager;
    $this->channelService = $channelService;
  }

  /**
   * Returns the number of registered providers.
   *
   * @return int
   *   The number of providers.
   */
  public function getProviderCount(): int {
    return count($this->providers);
  }

  /**
   * Returns the first available TV Channel node from providers.
   *
   * @param array $context
   *   Optional context for determining the channel.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The TV Channel node or NULL.
   */
  public function getChannelNode(array $context = []): ?NodeInterface {
    foreach ($this->providers as $provider) {
      $node = $provider->getChannelNode($context);
      if ($node instanceof NodeInterface) {
        return $node;
      }
    }
    return NULL;
  }

  /**
   * Returns the TV Channel nodes from providers.
   *
   * @param array $context
   *   Optional context for determining the channel.
   *
   * @return \Drupal\tv\Entity\Collection\TvChannelCollection
   *   The TV Channels, if any.
   */
  public function getChannels(array $context = []): TvChannelCollection {
    $collection = new TvChannelCollection($this->entityTypeManager, $this->channelService);
    foreach ($this->providers as $provider) {
      /** @var \Drupal\tv\Channel\ChannelContextProviderInterface $provider */
      $cids = $provider->getChannelIds($context);
      foreach ($cids as $cid) {
        $collection->append($cid);
      }
    }
    return $collection;
  }
}
