<?php

namespace Drupal\tv\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\node\Controller\NodeViewController;
use Drupal\tv\Service\TvChannelServiceInterface;
use Drupal\tv\Channel\ChannelContextCollector;
use Drupal\tv\Entity\Collection\TvChannelCollection;
use Psr\Container\ContainerInterface;

/**
 * Provides a 'TV' block.
 *
 * @Block(
 *   id = "tv",
 *   admin_label = @Translation("TV Block"),
 *   category = @Translation("Media")
 * )
 */
class TVBlock extends BlockBase implements ContainerFactoryPluginInterface {

    private TvChannelServiceInterface $tvChannelService;
    private RouteMatchInterface $routeMatch;
    private ChannelContextCollector $channelContextCollector;
    private EntityTypeManagerInterface $entityTypeManager;

    public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $routeMatch, TvChannelServiceInterface $tvChannelService, ChannelContextCollector $channelContextCollector)
    {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->tvChannelService = $tvChannelService;
        $this->routeMatch = $routeMatch;
        $this->channelContextCollector = $channelContextCollector;

        // @todo inject this
        $this->entityTypeManager = \Drupal::entityTypeManager();
    }

    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('current_route_match'),
            $container->get('tv.channel'),
            $container->get('tv.channel_context_collector')
        );
    }

    /**
     * {@inheritdoc}
     */
    public function build() {
        $build = [];

        $channels = $this->getChannels();
        if (!$channels->count()) {
            return $build;
        }

        // Don't show the TV if there are no channels with items.
        $itemCount = 0;
        foreach ($channels as $channel) {
            $itemCount += count($this->tvChannelService->getItems($channel));
        }
        if ($itemCount == 0) {
            return $build;
        }

        // Build attributes.
        $attributes['class'] = 'tv';
        $attributes['data-base-url'] = \Drupal::request()->getSchemeAndHttpHost();
        // @todo support multiple channels here, instead of using context-based channel computation via the channels controller.

        // Build the block.
        $build[] = [
            '#type' => 'container',
            '#attributes' => $attributes,
            '#attached' => [
                'library' => [
                    $this->isDev() ? 'tv/tv_dev' : 'tv/tv_prod',
                ],
            ],
        ];
        return $build;
    }

    /**
     * Check if development mode is enabled.
     *
     * Reads the 'tv_use_dev_build' setting from settings.php to determine
     * whether to load the development or production build of the React app.
     * Returns FALSE if the setting is not defined.
     *
     * To enable dev mode, add to settings.php:
     * $settings['tv_use_dev_build'] = TRUE;
     *
     * To disable dev mode, remove the line or set to FALSE.
     *
     * @return bool
     *   TRUE if development mode is enabled, FALSE otherwise.
     */
    private function isDev(): bool {
        return (bool) \Drupal\Core\Site\Settings::get('tv_use_dev_build', FALSE);
    }

    private function getChannels(): TvChannelCollection {
        // Try to get channels from context providers first, fallback to routeMatch.
        $channels = $this->channelContextCollector->getChannels();
        if ($channels->count() > 0) {
            return $channels;
        }

        $channels = new TvChannelCollection($this->entityTypeManager, $this->tvChannelService);
        $route = $this->routeMatch->getRouteObject();
        if (!$route) {
            throw new \Exception('No route available');
        }
        $controller = $route->getDefault('_controller');
        $node = $this->routeMatch->getParameter('node');
        if ($controller != '\\' . NodeViewController::class . '::view' || !$node) {
            throw new \Exception('Not a node view route');
        }
        $bundle = $node->bundle();
        if ($bundle != 'tv_channel') {
            return $channels;
        }
        $channels->append($node->id());
        return $channels;
    }

    public function getCacheTags() {
        if ($node = $this->routeMatch->getParameter('node')) {
            return Cache::mergeTags(parent::getCacheTags(), ['node:' . $node->id()]);
        } else {
            return parent::getCacheTags();
        }
    }

    public function getCacheContexts() {
        return Cache::mergeContexts(parent::getCacheContexts(), ['route']);
    }

    public function getCacheMaxAge() {
        return 0;
    }

}
