<?php

namespace Drupal\tv\Service;

use DateTime;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\file\Entity\File;
use Drupal\image\Entity\ImageStyle;
use Drupal\media\Entity\Media;
use Drupal\media\MediaInterface;
use Drupal\tv\ChannelItemProvider\TvChannelItemProviderCollector;
use Drupal\tv\ChannelItemSorter\TvChannelItemSorterInterface;
use Drupal\tv\Entity\TvChannel;
use Drupal\user\UserDataInterface;

class TvChannelService implements TvChannelServiceInterface {

    public function __construct(
        protected AccountProxyInterface $user,
        protected UserDataInterface $userData,
        protected TvChannelItemProviderCollector $providerCollector,
        protected TvChannelItemSorterInterface $itemSorter,
    ) {}

    /**
     * Formats a media entity into a channel item array.
     *
     * @param \Drupal\media\MediaInterface $video
     *   The media entity.
     *
     * @return array|null
     *   The formatted item array, or null if formatting fails.
     */
    protected function formatMediaItem(MediaInterface $video): ?array {
        $duration = $video->get('field_duration')->first()?->getValue()['value'] ?? null;
        $duration = is_numeric($duration) && $duration > 0 ? (int) $duration : 90;

        $urlField = $video->get('field_media_oembed_video')->first();
        if (!$urlField) {
            return null;
        }

        $item = [
            'id' => $video->id(),
            'name' => $video->getName(),
            'url' => $urlField->getValue()['value'],
            'createdDt' => (new DateTime())->setTimestamp($video->getCreatedTime())->format('c'),
            'startedDt' => $this->getStartedDateTime($video)?->format('c'),
            'startTs' => $this->getStartTimestamp($video),
            'endTs' => $this->getEndTimestamp($video),
            'duration' => $duration,
            'weight' => (int) ($video->get('field_weight')->value ?? 0),
        ];

        $image_media = $video->get('field_thumbnail')->entity ?? null;
        if ($image_media) {
            $managed_image_id = $image_media->get('field_media_image')->first()?->getValue()['target_id'] ?? null;
            if ($managed_image_id) {
                $file = File::load($managed_image_id);
                if ($file) {
                    $style = ImageStyle::load('thumbnail');
                    if ($style) {
                        $item['posterUrl'] = $style->buildUrl($file->getFileUri());
                    }
                }
            }
        }

        return $item;
    }

    public function getItems(TvChannel $channel): array {
        $channelItems = [];

        // Collect media IDs from all providers.
        $mediaIds = $this->providerCollector->collectMediaIds($channel);

        // Load and format media entities.
        $mediaEntities = Media::loadMultiple($mediaIds);
        foreach ($mediaEntities as $video) {
            $item = $this->formatMediaItem($video);
            if ($item !== null) {
                $channelItems[] = $item;
            }
        }

        // Always sort items before returning.
        return $this->itemSorter->sort($channelItems, $channel);
    }

    private function getStartedDateTime(Media $video): ?\DateTime
    {
        // @todo return when the user started watching the video.
        return NULL;
    }

    private function getCompletedDateTime(Media $video): ?\DateTime
    {
        // @todo return when the user completed watching the video.
        return NULL;
    }

    private function getStartTimestamp(Media $video): ?string
    {
        // For clips, return the field_in value (start timecode).
        // For regular videos, return "00:00:00".
        if ($video->bundle() === 'clip' && $video->hasField('field_in') && !$video->get('field_in')->isEmpty()) {
            return $video->get('field_in')->value;
        }
        return "00:00:00";
    }

    private function getEndTimestamp(Media $video): ?string
    {
        // For clips, return the field_out value (end timecode).
        if ($video->bundle() === 'clip' && $video->hasField('field_out') && !$video->get('field_out')->isEmpty()) {
          return $video->get('field_out')->value;
        }
        // For regular videos, return the timestamp for the last second.
        $duration = $video->get('field_duration')->first()?->getValue()['value'] ?? null;
        return $this->formatTimestamp($duration);
    }

    private function formatTimestamp(int $totalSeconds): string
    {
        // Use int math to calculate hours, minutes, and seconds.
        $hours = intdiv($totalSeconds, 3600);
        $minutes = intdiv($totalSeconds % 3600, 60);
        $seconds = $totalSeconds % 60;
        return sprintf('%02d:%02d:%02d', $hours, $minutes, $seconds);
    }

    public function getAutoPlayPreference(): bool
    {
        return $this->userData->get('tv', $this->user->id(), 'autoplay') ?? FALSE;
    }

    public function getMutePreference(): bool
    {
        return $this->userData->get('tv', $this->user->id(), 'mute') ?? TRUE;
    }
}
