/**
 * Keys - Keyboard and media key controller for TV app
 * 
 * @component
 * @scope Keyboard navigation and media key handling (Roku/Premiere style)
 * @responsibilities
 * - Handle all keyboard shortcuts (arrows, enter, space, etc)
 * - Support media keys (play/pause, prev/next, etc)
 * - Manage navigation between Channels/Playlist/Controls
 * - Build dynamic navigation map based on open accordion sections
 * - Emit key events via keysEvents emitter for Controls component
 * - Reset idle state on key press
 * @limits
 * - ❌ Must NOT directly manipulate player (emits events for Controls)
 * - ❌ Must NOT handle mouse events (Mouse.js handles this)
 * - ✅ Should ONLY handle keyboard input and navigation logic
 * @see .cursorrules for full architecture boundaries
 * 
 * @keys
 * - Up/Down/Left/Right: Navigate sections and items
 * - Enter/Return: Select item or toggle section
 * - Space: Play/Pause (always)
 * - J/L: Fine scrub back/forward
 * - Shift+Left/Right: Coarse scrub
 * - PageUp/PageDown: Channel up/down
 * - ,/. or MediaPrev/Next: Item prev/next
 * - F: Fullscreen
 * - M: Mute
 * - Home/End: First/last item
 * - Escape/Backspace: Back/close
 */
import { useEffect, useCallback } from 'react';
import tvStateMachine from '../tvStateMachine';

/**
 * Simple event emitter for key events
 * Used to communicate between Keys and Controls components
 */
class KeysEvents {
  constructor() {
    this.listeners = {};
  }
  on(event, cb) {
    if (!this.listeners[event]) this.listeners[event] = [];
    this.listeners[event].push(cb);
    return () => this.off(event, cb);
  }
  off(event, cb) {
    if (!this.listeners[event]) return;
    this.listeners[event] = this.listeners[event].filter(fn => fn !== cb);
  }
  emit(event, payload) {
    if (!this.listeners[event]) return;
    for (const cb of this.listeners[event]) cb(payload);
  }
}

const keysEvents = new KeysEvents();
export { keysEvents };

/**
 * Keys component - Keyboard controller with Roku-style navigation
 * @param {Object} props - Component props
 * @param {Array} props.channels - Array of channel objects
 * @param {Object} props.channel - Current channel object with playlist
 * @param {Function} props.setCurrentChannelId - Function to change channel
 * @param {Function} props.setCurrentItem - Function to change playlist item
 * @param {Function} props.setOpenIndex - Function to change open accordion section
 * @param {number} props.openIndex - Currently open accordion section index
 * @param {number} props.currentFocusIndex - Current keyboard focus index in navMap
 * @param {Function} props.setCurrentFocusIndex - Function to update focus index
 * @param {Object} props.tvRef - Reference to player for control commands
 * @param {Function} props.setIsIdle - Function to reset idle timer
 */
function Keys({ channels, channel, setCurrentChannelId, setCurrentItem, setOpenIndex, openIndex, currentFocusIndex, setCurrentFocusIndex, tvRef, setIsIdle }) {
  // Build navigation map dynamically
  const buildNavMap = useCallback(() => {
    const navMap = [];

    // Channels section
    navMap.push({ type: 'heading', section: 'channels', action: 'toggle' });
    if (openIndex === 0) { // Channels section is open
      channels.forEach((item, index) => {
        navMap.push({ type: 'item', section: 'channels', index, action: 'select', data: item });
      });
    }

    // Channel section
    navMap.push({ type: 'heading', section: 'channel', action: 'toggle' });
    if (openIndex === 1) { // Channel section is open
      (channel.playlist || []).forEach((item, index) => {
        navMap.push({ type: 'item', section: 'channel', index, action: 'select', data: item });
      });
    }

    // Controls
    for (let i = 0; i < 3; i++) {
      navMap.push({ type: 'control', index: i, action: 'activate' });
    }

    return navMap;
  }, [channels, channel, openIndex]);

  const navMap = buildNavMap();

  const handleKeyDown = useCallback((e) => {
    keysEvents.emit('keydown', e);
    setIsIdle(false); // Reset idle state on key press
    // Navigation
    const current = navMap[currentFocusIndex];
    switch (e.key) {
        case 'ArrowUp':
          e.preventDefault();
          if (current && current.type !== 'control') {
            setCurrentFocusIndex(prev => Math.max(0, prev - 1));
          }
          break;
        case 'ArrowDown':
          e.preventDefault();
          if (current && current.type !== 'control') {
            const maxAccordionIdx = navMap.findLastIndex(item => item.type !== 'control');
            setCurrentFocusIndex(prev => Math.min(maxAccordionIdx, prev + 1));
          }
          break;
        case 'ArrowLeft':
          e.preventDefault();
          if (current && current.type === 'control') {
            // Move left in controls
            const controlIndices = navMap.map((item, idx) => item.type === 'control' ? idx : null).filter(idx => idx !== null);
            const currentControlIdx = controlIndices.indexOf(currentFocusIndex);
            if (currentControlIdx > 0) {
              setCurrentFocusIndex(controlIndices[currentControlIdx - 1]);
            }
          } else {
            // From accordion, go to controls (right-most)
            const lastControlIdx = navMap.findLastIndex(item => item.type === 'control');
            if (lastControlIdx !== -1) {
              setCurrentFocusIndex(lastControlIdx);
            }
          }
          break;
        case 'ArrowRight':
          e.preventDefault();
          if (current && current.type === 'control') {
            // Move right in controls, or go to accordion if last
            const controlIndices = navMap.map((item, idx) => item.type === 'control' ? idx : null).filter(idx => idx !== null);
            const currentControlIdx = controlIndices.indexOf(currentFocusIndex);
            if (currentControlIdx < controlIndices.length - 1) {
              setCurrentFocusIndex(controlIndices[currentControlIdx + 1]);
            } else {
              // Go to accordion
              const firstAccordionIdx = navMap.findIndex(item => item.type !== 'control');
              if (firstAccordionIdx !== -1) {
                setCurrentFocusIndex(firstAccordionIdx);
              }
            }
          }
          break;
        case 'Enter':
        case 'NumpadEnter':
        case 'Return':
          e.preventDefault();
          const currentItem = navMap[currentFocusIndex];
          if (currentItem) {
            if (currentItem.action === 'toggle') {
              if (currentItem.section === 'channels') {
                setOpenIndex(prev => prev === 0 ? -1 : 0);
              } else if (currentItem.section === 'channel') {
                setOpenIndex(prev => prev === 1 ? -1 : 1);
              }
            } else if (currentItem.action === 'select') {
              if (currentItem.section === 'channels') {
                setCurrentChannelId(currentItem.data.id);
                setOpenIndex(1);
              } else if (currentItem.section === 'channel') {
                // Check if video is fully played (progress >= 95%) and reset if needed
                const video = currentItem.data;
                if (video && video.progress >= 95) {
                  // Reset progress for fully played videos so they can be played from start
                  const baseUrl = window.baseUrl || '';
                  fetch(baseUrl + '/tv/progress', {
                    method: 'POST',
                    headers: {'Content-Type': 'application/json'},
                    body: JSON.stringify({media_id: video.id, progress: 0})
                  }).catch(e => console.error('Progress reset failed:', e));
                }
                tvStateMachine.transition('PLAY');
                setCurrentItem(currentItem.index);
              }
            } else if (currentItem.action === 'activate') {
              keysEvents.emit('control_select', currentItem.index);
            }
          }
          break;
        // Play/Pause (spacebar always toggles)
        case ' ':
        case 'Spacebar':
        case 'MediaPlayPause':
          tvStateMachine.transition('PLAY_PAUSE');
          keysEvents.emit('play_pause', e);
          break;
        // Fine scrubbing (Premiere style)
        case 'j':
          keysEvents.emit('scrub_fine_back', e);
          break;
        case 'l':
          keysEvents.emit('scrub_fine_forward', e);
          break;
        // Channel up/down
        case 'PageUp':
        case 'ChannelUp':
          keysEvents.emit('channel_up', e);
          break;
        case 'PageDown':
        case 'ChannelDown':
          keysEvents.emit('channel_down', e);
          break;
        // Item prev/next
        case ',':
        case 'MediaPreviousTrack':
          keysEvents.emit('item_prev', e);
          break;
        case '.':
        case 'MediaNextTrack':
          keysEvents.emit('item_next', e);
          break;
        // Scrub +30s / -30s
        case '=':
        case '+':
          keysEvents.emit('scrub_coarse_forward', e);
          break;
        case '-':
        case '_':
          keysEvents.emit('scrub_coarse_back', e);
          break;
        // Home/End
        case 'Home':
          keysEvents.emit('first', e);
          break;
        case 'End':
          keysEvents.emit('last', e);
          break;
        // Fullscreen
        case 'f':
        case 'F':
          keysEvents.emit('fullscreen_toggle', e);
          break;
        // Mute (no remember)
        case 'm':
        case 'M':
          keysEvents.emit('mute_toggle', e);
          break;
        // Back/close
        case 'Backspace':
        case 'Escape':
          keysEvents.emit('back', e);
          break;
        default:
          break;
      }
      // Media key support (for browsers that use e.code)
      switch (e.code) {
        case 'MediaTrackPrevious':
          keysEvents.emit('item_prev', e);
          break;
        case 'MediaTrackNext':
          keysEvents.emit('item_next', e);
          break;
        case 'MediaPlayPause':
          keysEvents.emit('play_pause', e);
          break;
        case 'MediaStop':
          keysEvents.emit('stop', e);
          break;
        case 'MediaFastForward':
          keysEvents.emit('scrub_coarse_forward', e);
          break;
        case 'MediaRewind':
          keysEvents.emit('scrub_coarse_back', e);
          break;
        default:
          break;
      }
    }, [navMap, currentFocusIndex, setCurrentFocusIndex, setOpenIndex, setCurrentChannelId, setCurrentItem, setIsIdle]);

  useEffect(() => {
    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [handleKeyDown]);
  return null;
}

export default Keys;
