/**
 * playbackEvents - Progress tracking utilities
 * 
 * @module
 * @scope Progress persistence and loading utilities
 * @responsibilities
 * - Update progress in localStorage and channel state
 * - Load progress from localStorage for playlist items
 * - Provide progress update callback factory
 * @limits
 * - ❌ Does NOT calculate progress percentages (Player.js does this)
 * - ❌ Does NOT handle clip boundaries (Player.js does this)
 * - ✅ Should ONLY handle localStorage read/write and state updates
 * @see .cursorrules for full architecture boundaries
 * @deprecated Consider consolidating into useProgress hook (see TODO #2 in .cursorrules)
 */

import { useEffect } from 'react';

/** Prefix for localStorage progress keys */
const PROGRESS_KEY_PREFIX = 'tv_progress_';

/**
 * Update progress for a playlist item
 * Factory function that creates progress updater for channel
 * @param {Object} channel - Channel object with playlist
 * @param {Function} setChannel - Function to update channel state
 * @returns {Function} Progress updater (itemId, newProgress) => void
 */
export const updateProgress = (channel, setChannel) => (itemId, newProgress) => {
  const item = channel.playlist.find(i => i.id === itemId);
  if (item) {
    // Store in localStorage
    try {
      localStorage.setItem(PROGRESS_KEY_PREFIX + itemId, String(newProgress));
    } catch (e) {
      // Fallback: ignore if localStorage is unavailable
    }
    setChannel(prevChannel => {
      const updatedPlaylist = prevChannel.playlist.map(item => {
        if (item.id === itemId) {
          return { ...item, progress: newProgress };
        }
        return item;
      });
      return { ...prevChannel, playlist: updatedPlaylist };
    });
  }
};

/**
 * Load progress for all items in a playlist from localStorage
 * @param {Array} playlist - Array of playlist items
 * @returns {Array} Playlist with progress loaded from localStorage
 */
export function loadProgressForPlaylist(playlist) {
  return playlist.map(item => {
    if (!item || !item.id) return item;
    let progress = item.progress;
    try {
      const stored = localStorage.getItem(PROGRESS_KEY_PREFIX + item.id);
      if (stored !== null) {
        progress = parseInt(stored, 10);
        if (isNaN(progress)) progress = 0;
      }
    } catch (e) {
      // Ignore localStorage errors
    }
    return { ...item, progress };
  });
}

/**
 * Custom hook to poll player state and update React state
 * @param {Object} tvRef - Reference to player
 * @param {string} playerState - Current player state
 * @param {Function} setPlayerState - Function to update player state
 * @deprecated Polling approach may be replaced by event-driven state updates
 */
export function usePlayerState(tvRef, playerState, setPlayerState) {
  useEffect(() => {
    const interval = setInterval(() => {
      if (
        tvRef.current &&
        tvRef.current.defaultPlayer &&
        typeof tvRef.current.defaultPlayer.getPlayerState === 'function'
      ) {
        const state = tvRef.current.defaultPlayer.getPlayerState();
        if (state === 1) {
          if (playerState !== 'playing') setPlayerState('playing');
        } else {
          if (playerState !== 'paused') setPlayerState('paused');
        }
      }
    }, 300);
    return () => clearInterval(interval);
  }, [tvRef, playerState, setPlayerState]);
}
