/**
 * Unit tests for Accordion component
 * Tests accordion container that manages multiple AccordionMenu children
 */

import React from 'react';
import { render, screen, fireEvent } from '@testing-library/react';
import Accordion from '../Components/Accordion';
import AccordionMenu from '../Components/AccordionMenu';

describe('Accordion', () => {
  test('renders children', () => {
    render(
      <Accordion>
        <AccordionMenu heading="Section 1">
          <div>Section 1 content</div>
        </AccordionMenu>
        <AccordionMenu heading="Section 2">
          <div>Section 2 content</div>
        </AccordionMenu>
      </Accordion>
    );

    expect(screen.getByText('Section 1')).toBeInTheDocument();
    expect(screen.getByText('Section 2')).toBeInTheDocument();
  });

  test('opens first section by default when defaultOpenIndex=0', () => {
    render(
      <Accordion defaultOpenIndex={0}>
        <AccordionMenu heading="Section 1">
          <div>Section 1 content</div>
        </AccordionMenu>
        <AccordionMenu heading="Section 2">
          <div>Section 2 content</div>
        </AccordionMenu>
      </Accordion>
    );

    // First section content should be visible
    expect(screen.getByText('Section 1 content')).toBeInTheDocument();
  });

  test('handles controlled openIndex prop', () => {
    const setOpenIndex = jest.fn();

    render(
      <Accordion openIndex={0} setOpenIndex={setOpenIndex}>
        <AccordionMenu heading="Section 1">
          <div>Section 1 content</div>
        </AccordionMenu>
        <AccordionMenu heading="Section 2">
          <div>Section 2 content</div>
        </AccordionMenu>
      </Accordion>
    );

    // Click section 2 heading
    const heading2 = screen.getByText('Section 2');
    fireEvent.click(heading2);

    expect(setOpenIndex).toHaveBeenCalledWith(1);
  });

  test('handles empty children', () => {
    render(<Accordion />);

    // Should render without crashing
    expect(screen.queryByRole('heading')).not.toBeInTheDocument();
  });

  test('injects props into AccordionMenu children', () => {
    const { container } = render(
      <Accordion openIndex={0}>
        <AccordionMenu heading="Section 1">
          <div>Content</div>
        </AccordionMenu>
      </Accordion>
    );

    // AccordionMenu should receive open prop and render accordingly
    expect(container.querySelector('.accordion-menu')).toBeInTheDocument();
  });
});
