/**
 * AccordionMenu - Generic accordion menu section with dynamic scrollable height
 * 
 * @component
 * @scope Reusable UI component for expandable/collapsible menu sections
 * @responsibilities
 * - Render expandable heading with chevron indicator
 * - Show/hide content based on open state
 * - Handle dynamic scrollable height from parent Accordion
 * - Apply focus styling when keyboard navigation active
 * @limits
 * - ❌ Must NOT manage open state internally (parent Accordion controls this)
 * - ❌ Must NOT fetch data or make API calls
 * - ✅ Should be pure presentation component
 * @see .cursorrules for full architecture boundaries
 */
import './AccordionMenu.css';

/**
 * AccordionMenu component - Single accordion section
 * @param {Object} props - Component props
 * @param {boolean} props.open - Whether section is expanded
 * @param {Function} props.setOpen - Function to toggle section (called with no args)
 * @param {string|ReactNode} props.heading - Heading text or element
 * @param {ReactNode} props.children - Content to show when expanded
 * @param {string} props.className - Optional CSS class name
 * @param {Object} props.headingRef - Ref for heading element (for height calculation)
 * @param {boolean} props.focused - Whether this section is keyboard focused
 * @param {number} props.scrollableHeight - Height in pixels for scrollable content area
 */
function AccordionMenu({ open, setOpen, heading, children, className = '', headingRef, focused, scrollableHeight }) {
  const handleClick = () => {
    if (setOpen) {
      setOpen();
    } else {
      console.error('[AccordionMenu] setOpen function is not provided');
    }
  };

  return (
    <div className={`accordion-menu ${className}`.trim()}>
      <h3 ref={headingRef} onClick={handleClick} className={focused ? 'focused' : ''} style={{ cursor: 'pointer', userSelect: 'none' }}>
        {heading} {open ? <span style={{fontWeight:400}}>&#9660;</span> : <span style={{fontWeight:400}}>&#9654;</span>}
      </h3>
      <div
        className="scrollable"
        style={open ? { display: 'block', ...(scrollableHeight ? { height: scrollableHeight, flex: 'none' } : { flex: 1 }) } : { display: 'none' }}
      >
        {children}
      </div>
    </div>
  );
}

export default AccordionMenu;
