/**
 * Player - Playback coordinator between state machine and provider wrappers
 * 
 * @component
 * @scope Coordinates playback, manages clip boundaries, calculates resume positions
 * @responsibilities
 * - Calculate resume positions from localStorage progress
 * - Manage clip boundaries (startTs/endTs) for video segments
 * - Poll current time and update progress in localStorage
 * - Auto-advance when reaching endTs
 * - Expose seekTo() and getCurrentTime() via ref
 * @limits
 * - ❌ Must NOT contain provider-specific API calls (use wrappers like YouTube.js)
 * - ❌ Must NOT render UI (returns provider component only)
 * - ❌ Must NOT handle keyboard/mouse events (Controls.js handles this)
 * @see .cursorrules for full architecture boundaries
 */
import React, { forwardRef, useEffect, useRef, useMemo } from "react";
import YouTube from "./Embed/YouTube/YouTube";
import { parseDuration, getInSeconds } from "../../helpers";
const parseYouTubeDuration = (duration) => {
    if (!duration || typeof duration !== 'string') return 0;
    const match = duration.match(/PT(?:(\d+)H)?(?:(\d+)M)?(?:(\d+)S)?/);
    if (!match) return 0;
    const hours = parseInt(match[1] || 0, 10);
    const minutes = parseInt(match[2] || 0, 10);
    const seconds = parseInt(match[3] || 0, 10);
    return hours * 3600 + minutes * 60 + seconds;
};

/**
 * Player component - Coordinates playback between state machine and provider
 * @param {Object} props - Component props
 * @param {Object} props.item - Current playlist item with url, duration, startTs, endTs, id
 * @param {Object} props.channel - Channel object
 * @param {Function} props.onVideoEnd - Callback when video/clip ends
 * @param {Function} props.updateProgress - Callback to update progress in parent
 * @param {Object} ref - Forwarded ref exposing seekTo, getCurrentTime, desiredStartTimeRef
 */
const Player = forwardRef(function Player(params, ref) {
    const { item, channel, onVideoEnd, ...rest } = params;
    const pollingRef = useRef();
    const innerRef = useRef();
    const playerRef = ref ? ref : innerRef;
    const hasSeekedRef = useRef(false);
    const lastItemIdRef = useRef(null);
    const itemLoadTimeRef = useRef(null);
    const desiredStartTimeRef = useRef(null);

    // Also expose desiredStartTimeRef directly on playerRef for access from outside
    useEffect(() => {
        if (playerRef.current) {
            playerRef.current.desiredStartTimeRef = desiredStartTimeRef;
        }
    }, [playerRef]);

    // Expose seekTo for progress bar scrubbing
    React.useImperativeHandle(ref, () => ({
        seekTo: (seconds, play = true) => {
            if (playerRef.current && playerRef.current.seekTo) {
                playerRef.current.seekTo(Number(seconds), play);
            }
        },
        getCurrentTime: () => {
            if (playerRef.current && playerRef.current.getCurrentTime) {
                return playerRef.current.getCurrentTime();
            }
            return 0;
        },
        // Expose desiredStartTimeRef for progress bar seeking
        desiredStartTimeRef: desiredStartTimeRef,
        triggerSeekFromProgressBar: () => {
            const seekTime = desiredStartTimeRef.current;
            if (seekTime != null && playerRef.current?.players?.youtube) {
                try {
                    playerRef.current.players.youtube.seekTo(seekTime, true);
                    desiredStartTimeRef.current = null;
                } catch (e) {
                }
            }
        }
    }), [playerRef]);

    // Extract dependencies for useEffect
    const itemEndTs = item && item.endTs;
    const startTsSeconds = item && item.startTs ? getInSeconds(item.startTs) : 0;

    // Calculate startSeconds from saved progress
    const startSeconds = useMemo(() => {
        if (!item || !item.id) return startTsSeconds;
        let savedProgressPercent = 0;
        try {
            const stored = localStorage.getItem('tv_progress_' + item.id);
            if (stored) {
                const parsed = JSON.parse(stored);
                if (parsed && typeof parsed === 'object' && parsed.percent) {
                    savedProgressPercent = parsed.percent;
                } else if (!isNaN(Number(stored))) {
                    savedProgressPercent = Number(stored);
                }
            }
        } catch (e) {}
        if (savedProgressPercent > 0 && savedProgressPercent < 95) {
            const duration = typeof item.duration === 'number' ? item.duration : parseDuration(item.duration);
            const clipStart = startTsSeconds;
            const clipEnd = item.endTs ? getInSeconds(item.endTs) : duration;
            const clipLength = clipEnd - clipStart;
            if (clipLength > 0) {
                const resumeSeconds = clipStart + Math.floor((savedProgressPercent * clipLength) / 100);
                const finalResumePosition = Math.max(clipStart, Math.min(clipEnd - 1, resumeSeconds));
                return finalResumePosition;
            }
        }
        return startTsSeconds;
    }, [item, startTsSeconds]);

    // Reset seek flag when item changes
    useEffect(() => {
        const itemId = item?.id || item?.url;
        if (itemId && itemId !== lastItemIdRef.current) {
            hasSeekedRef.current = false;
            lastItemIdRef.current = itemId;
            itemLoadTimeRef.current = Date.now();
        }
    }, [item]);

    // Monitor player state and seek to saved position once playback begins
    // DISABLED: Now handled in YouTube.js onReady callback with cueVideoById
    /*
    useEffect(() => {
        if (!item || !playerRef.current) return;

        const pollForPlaying = setInterval(() => {
            if (hasSeekedRef.current) {
                clearInterval(pollForPlaying);
                return;
            }

            const player = playerRef.current?.players?.youtube;
            if (player && typeof player.getPlayerState === 'function') {
                const state = player.getPlayerState();
                // State 1 = playing
                if (state === 1 && !hasSeekedRef.current) {
                    hasSeekedRef.current = true;
                    const seekTo = item.savedResumePosition || startSeconds;
                    if (seekTo > 0) {
                        log('Playback started, seeking to position:', seekTo);
                        setTimeout(() => {
                            try {
                                player.seekTo(seekTo, true);
                            } catch (e) {
                                log('Error seeking:', e);
                            }
                        }, 100);
                    }
                    clearInterval(pollForPlaying);
                }
            }
        }, 250);

        return () => clearInterval(pollForPlaying);
    }, [item, playerRef, startSeconds]);
    */

    // Poll for endTs and auto-advance, plus track progress
    useEffect(() => {
        if (!item || !item.url || !playerRef.current) return;
        if (pollingRef.current) clearInterval(pollingRef.current);
        const endTs = item.endTs ? getInSeconds(item.endTs) : null;
        const startTs = item.startTs ? getInSeconds(item.startTs) : 0;
        // Add validation for timestamps
        if (endTs !== null && endTs <= startTs) {
            return;
        }
        const duration = typeof item.duration === 'number' ? item.duration : parseYouTubeDuration(item.duration);
        const clipEnd = endTs !== null ? endTs : duration;
        const clipLength = clipEnd - startTs;

        pollingRef.current = setInterval(() => {
            if (!playerRef.current || !playerRef.current.getCurrentTime) return;
            const current = playerRef.current.getCurrentTime();

            // Update progress in localStorage
            // Delay progress updates for 3 seconds after item load to prevent overriding saved progress
            // Also don't update if current time is 0 (video not loaded yet)
            const timeSinceLoad = itemLoadTimeRef.current ? Date.now() - itemLoadTimeRef.current : 0;
            if (item.id && clipLength > 0 && current >= startTs && current > 0 && timeSinceLoad > 3000) {
                const progressPercent = Math.floor(((current - startTs) * 100) / clipLength);
                const clampedPercent = Math.max(0, Math.min(100, progressPercent));
                try {
                    localStorage.setItem('tv_progress_' + item.id, String(clampedPercent));
                } catch (e) {}
                // Also call updateProgress if provided
                if (typeof params.updateProgress === 'function') {
                    params.updateProgress(item.id, clampedPercent);
                }
            }

            // Check for endTs and auto-advance
            if (endTs != null && current >= endTs) {
                if (playerRef.current.pause) playerRef.current.pause();
                if (typeof onVideoEnd === 'function') onVideoEnd();
                clearInterval(pollingRef.current);
            }
        }, 500);
        return () => { if (pollingRef.current) clearInterval(pollingRef.current); };
    }, [item, itemEndTs, onVideoEnd, playerRef, params]);

    return (
        <YouTube
            {...rest}
            item={item}
            channel={channel}
            tvRef={playerRef}
            startSeconds={startSeconds}
            onError={params.onError}
            onPlayerStateChange={params.onPlayerStateChange}
        />
    );
});

export default Player;
