/**
 * Unit tests for ErrorBoundary component
 * Tests error catching and fallback UI rendering
 */

import React from 'react';
import { render, screen } from '@testing-library/react';
import ErrorBoundary from '../Components/ErrorBoundary';

// Component that throws an error
const ThrowError = ({ shouldThrow }) => {
  if (shouldThrow) {
    throw new Error('Test error');
  }
  return <div>No error</div>;
};

// Component that throws on render
const AlwaysThrows = () => {
  throw new Error('Always throws error');
};

describe('ErrorBoundary', () => {
  // Suppress console.error during tests
  beforeEach(() => {
    jest.spyOn(console, 'error').mockImplementation(() => {});
  });

  afterEach(() => {
    console.error.mockRestore();
  });

  test('renders children when no error occurs', () => {
    render(
      <ErrorBoundary>
        <div>Child component</div>
      </ErrorBoundary>
    );

    expect(screen.getByText('Child component')).toBeInTheDocument();
  });

  test('catches error and displays fallback UI', () => {
    render(
      <ErrorBoundary>
        <AlwaysThrows />
      </ErrorBoundary>
    );

    // ErrorBoundary should show error message instead of children
    expect(screen.queryByText('Always throws error')).not.toBeInTheDocument();
    // Check for fallback UI elements (based on ErrorBoundary implementation)
    expect(screen.getByText('Something went wrong')).toBeInTheDocument();
  });

  test('displays error details in fallback UI', () => {
    render(
      <ErrorBoundary>
        <AlwaysThrows />
      </ErrorBoundary>
    );

    // Should display the error message in details section
    expect(screen.getByText('Error Details')).toBeInTheDocument();
    // Error text appears multiple times (error message, stack, etc)
    const errorTexts = screen.getAllByText(/Always throws error/);
    expect(errorTexts.length).toBeGreaterThan(0);
  });

  test('does not catch errors in event handlers', () => {
    // ErrorBoundary only catches errors in render, lifecycle, constructors
    // Not in event handlers, async code, or server-side rendering

    const handleClick = () => {
      throw new Error('Event handler error');
    };

    const ComponentWithHandler = () => (
      <button onClick={handleClick}>Click me</button>
    );

    // This should not trigger ErrorBoundary
    expect(() => {
      render(
        <ErrorBoundary>
          <ComponentWithHandler />
        </ErrorBoundary>
      );
    }).not.toThrow();
  });

  test('can recover by rendering different children', () => {
    const { rerender } = render(
      <ErrorBoundary>
        <ThrowError shouldThrow={true} />
      </ErrorBoundary>
    );

    // Error boundary should show fallback
    expect(screen.getByText('Something went wrong')).toBeInTheDocument();

    // Rerender with non-throwing component
    rerender(
      <ErrorBoundary>
        <ThrowError shouldThrow={false} />
      </ErrorBoundary>
    );

    // Note: ErrorBoundary might need a key prop or reset mechanism to recover
    // This test documents expected recovery behavior
  });

  test('logs error to console', () => {
    render(
      <ErrorBoundary>
        <AlwaysThrows />
      </ErrorBoundary>
    );

    // console.error should be called with error and component stack
    expect(console.error).toHaveBeenCalled();
  });

  test('works with nested error boundaries', () => {
    const InnerError = () => {
      throw new Error('Inner error');
    };

    render(
      <ErrorBoundary>
        <div>Outer content</div>
        <ErrorBoundary>
          <InnerError />
        </ErrorBoundary>
      </ErrorBoundary>
    );

    // Inner error boundary should catch inner error and show error UI
    // The outer content div is still rendered alongside the inner error boundary
    expect(screen.getByText('Outer content')).toBeInTheDocument();
    // Inner error boundary shows "Something went wrong"
    expect(screen.getByText('Something went wrong')).toBeInTheDocument();
    // Error details contain the inner error message
    const errorTexts = screen.getAllByText(/Inner error/);
    expect(errorTexts.length).toBeGreaterThan(0);
  });
});
