/**
 * tvStateMachine - Global state management for TV app
 * 
 * @module
 * @scope Centralized state management - single source of truth for player and UI state
 * @responsibilities
 * - Maintain state: playerState, currentItem, playlistLength, userIntent, fullscreen, overlay
 * - Provide transition() method for state changes
 * - Provide subscribe() method for state listeners
 * - Calculate first/next/previous qualifying items (progress < 95%, remaining >= 30s)
 * @limits
 * - ❌ Must NOT contain React code
 * - ❌ Must NOT directly manipulate DOM or localStorage
 * - ❌ Must NOT make API calls
 * - ✅ Should be pure state management only
 * - ✅ React components subscribe and sync FROM state machine
 * @see .cursorrules for full architecture boundaries
 */
import { parseDuration } from './helpers';

/**
 * Returns the index of the first item with progress < 95% (remaining >= 30s), or 0 if none found
 * @param {Array} playlistArg - Playlist array
 * @returns {number} Index of first qualifying item
 */
function getFirstQualifyingItem(playlistArg) {
  let playlist = playlistArg || [];
  if (!playlist.length) return 0;
  for (let idx = 0; idx < playlist.length; idx++) {
    const item = playlist[idx];
    if (item) {
      let duration = item.duration;
      if (typeof duration === 'string' && duration.startsWith('PT')) {
        duration = parseDuration(duration);
      }
      duration = Number(duration) || 0;
      let watched = Number(item.progress) || 0;
      watched = (watched / 100) * duration;
      const remaining = duration - watched;
      if (remaining >= 30) return idx;
    }
  }
  return 0; // Default to first item if all are completed
}

/**
 * Returns the index of the next item with progress < 95%, or -1 if none found
 * @param {Array} playlistArg - Playlist array
 * @returns {number} Index of next qualifying item, or -1 if none
 */
function getNextQualifyingItem(playlistArg) {
  let playlist = playlistArg || [];
  let playlistLength = playlist.length;
  let current = state.currentItem;
  if (!playlistLength) return -1;
  for (let idx = current + 1; idx < playlistLength; idx++) {
    const item = playlist[idx];
    if (item) {
      let duration = item.duration;
      if (typeof duration === 'string' && duration.startsWith('PT')) {
        duration = parseDuration(duration);
      }
      duration = Number(duration) || 0;
      let watched = Number(item.progress) || 0;
      watched = (watched / 100) * duration;
      const remaining = duration - watched;
      if (remaining >= 30) return idx;
    }
  }
  return -1;
}

/**
 * Returns the index of the previous item with progress < 95%, or current item if none found
 * @returns {number} Index of previous qualifying item, or current item
 */
function getPrevQualifyingItem() {
  let playlist = [];
  let playlistLength = state.playlistLength;
  let current = state.currentItem;
  if (typeof window !== 'undefined' && window.tvChannel && Array.isArray(window.tvChannel.playlist)) {
    playlist = window.tvChannel.playlist;
    playlistLength = playlist.length;
  }
  if (!playlistLength) return -1;
  for (let idx = current - 1; idx >= 0; idx--) {
    const item = playlist[idx];
    if (item) {
      let duration = item.duration;
      if (typeof duration === 'string' && duration.startsWith('PT')) {
        duration = parseDuration(duration);
      }
      duration = Number(duration) || 0;
      let watched = Number(item.progress) || 0;
      watched = (watched / 100) * duration;
      if ((duration - watched) >= 30) return idx;
    }
  }
  return current;
}

const initialContext = {
  overlay: false,
  fullscreen: false,
  playerState: 'idle',   // 'idle' | 'playing' | 'paused' | 'ended' | 'loading' | 'error'
  userIntent: null,      // 'navigate' | 'scrub' | 'select' | 'idle' | null
  currentItem: 0,        // index of the current playlist item
  playlistLength: 0,     // length of the current playlist (set by payload)
};

let state = {
  ...initialContext,
};

const listeners = [];

function notify(event, prev, next) {
  // Explicit logging for state machine notifications
  listeners.forEach(fn => fn(event, prev, next));
}

function transition(event, payload) {
  // Explicit logging for state machine transitions
  const prev = { ...state };
  let changed = false;
  switch (event) {
        case 'SET_PLAYLIST_LENGTH':
          if (typeof payload === 'number') {
            state.playlistLength = payload;
          }
          break;
        case 'SET_CURRENT_ITEM':
          if (typeof payload === 'number') {
            state.currentItem = payload;
          }
          break;
        case 'NEXT_ITEM': {
          if (state.playlistLength > 0) {
            // Accept playlist as payload for up-to-date progress
            const playlist = payload && payload.playlist ? payload.playlist : undefined;
            const nextQualifyingItem = getNextQualifyingItem(playlist);
            if (nextQualifyingItem !== -1 && nextQualifyingItem !== state.currentItem) {
              state.currentItem = nextQualifyingItem;
              state.userIntent = 'navigate';
              state.playerState = 'playing';
              changed = true;
            } else {
              // No next item, mark as ended
              state.playerState = 'ended';
              state.userIntent = 'idle';
              changed = true;
            }
          }
          break;
        }
    case 'PAUSE':
      if (state.playerState !== 'paused') {
        state.playerState = 'paused';
        state.userIntent = 'pause';
        changed = true;
      }
      break;
    case 'MUTE':
      state.userIntent = 'mute';
      changed = true;
      break;
    case 'UNMUTE':
      state.userIntent = 'unmute';
      changed = true;
      break;
    case 'END': {
      // Only update state to ended; do not mutate playlist or progress or restart playback.
      state.playerState = 'ended';
      state.userIntent = 'idle';
      changed = true;
      // eslint-disable-next-line no-console
      break;
    }
    case 'SCRUB':
      state.userIntent = 'scrub';
      break;
    case 'FULLSCREEN_ENTER':
      state.fullscreen = true;
      break;
    case 'FULLSCREEN_EXIT':
      state.fullscreen = false;
      break;
    case 'TOGGLE_CC':
      state.ccEnabled = !state.ccEnabled;
      changed = true;
      break;
    case 'ERROR':
      state.playerState = 'error';
      state.userIntent = 'idle';
      break;
    case 'RESET':
      state = { ...initialContext };
      break;
    case 'PLAY':
      if (state.playerState === 'idle' || state.playerState === 'paused') {
        state.playerState = 'playing';
        state.userIntent = 'play';
        changed = true;
      }
      break;
    default:
      // Allow custom context updates
      if (payload && typeof payload === 'object') {
        state = { ...state, ...payload };
      }
      break;
  }
  if (changed || event === 'PLAY' || event === 'PAUSE' || event === 'MUTE' || event === 'UNMUTE') {
    // ...removed debug log...
    notify(event, prev, { ...state });
    // ...removed debug log...
  }
}

function getState() {
  return { ...state };
}

function subscribe(fn) {
  listeners.push(fn);
  return () => {
    const idx = listeners.indexOf(fn);
    if (idx !== -1) listeners.splice(idx, 1);
  };
}

const tvStateMachine = {
  transition,
  getState,
  subscribe,
  getFirstQualifyingItem,
  getNextQualifyingItem,
  getPrevQualifyingItem,
};

// Expose globally for Controls and other components
if (typeof window !== 'undefined') {
  window.tvStateMachine = tvStateMachine;
}

export default tvStateMachine;
