/**
 * Accordion - Container managing open/close state for multiple AccordionMenu children
 *
 * @component
 * @scope Container component managing accordion state and dynamic height calculations
 * @responsibilities
 * - Manage which AccordionMenu is currently open (only one at a time)
 * - Calculate and distribute scrollable heights to children
 * - Inject open/setOpen/scrollableHeight props into AccordionMenu children
 * - Handle window resize events to recalculate heights
 * @limits
 * - ❌ Must NOT render content directly (uses children pattern)
 * - ❌ Must NOT fetch data or handle business logic
 * - ✅ Should ONLY manage accordion UI state
 * @see .cursorrules for full architecture boundaries
 */
import React from 'react';

import './Accordion.css';

import { useRef, useLayoutEffect, useState, useMemo } from 'react';

/**
 * Accordion component - Manages multiple AccordionMenu children
 * @param {Object} props - Component props
 * @param {ReactNode} props.children - AccordionMenu components
 * @param {number} props.openIndex - Controlled: which section is open (0-based index)
 * @param {Function} props.setOpenIndex - Controlled: function to set open section
 * @param {number} props.defaultOpenIndex - Uncontrolled: initial open section (default 0)
 * @example
 * <Accordion>
 *   <AccordionMenu heading="Channels">...</AccordionMenu>
 *   <AccordionMenu heading="Playlist">...</AccordionMenu>
 * </Accordion>
 */
function Accordion({ children, openIndex, setOpenIndex, defaultOpenIndex = 0 }) {
  const [internalOpenIndex, setInternalOpenIndex] = React.useState(defaultOpenIndex);
  const currentOpenIndex = openIndex !== undefined ? openIndex : internalOpenIndex;
  const currentSetOpenIndex = setOpenIndex || setInternalOpenIndex;
  const containerRef = useRef(null);
  const [containerHeight, setContainerHeight] = useState(undefined);
  const [headingHeights, setHeadingHeights] = useState([]);

  // Gather heading refs from children - persist across renders
  const headingRefs = useMemo(() =>
    React.Children.map(children, () => React.createRef()) || [],
    [children]
  );

  useLayoutEffect(() => {
    function updateHeights() {
      if (containerRef.current) {
        setContainerHeight(containerRef.current.parentElement.offsetHeight);
        setHeadingHeights(
          headingRefs.map(ref => ref.current ? ref.current.offsetHeight : 0)
        );
      }
    }
    updateHeights();
    window.addEventListener('resize', updateHeights);
    return () => window.removeEventListener('resize', updateHeights);
  }, [children, headingRefs]);

  // Clone children and inject open/setOpen/headingRef/scrollableHeight props
  const menus = React.Children.map(children, (child, idx) => {
    if (!React.isValidElement(child)) return child;
    const isOpen = currentOpenIndex === idx;
    let scrollableHeight = undefined;
    if (containerHeight && headingHeights.length > 0) {
      const totalHeadingsHeight = headingHeights.reduce((sum, h) => sum + h, 0);
      scrollableHeight = Math.max(containerHeight - totalHeadingsHeight, 0);
    }
    return React.cloneElement(child, {
      open: isOpen,
      setOpen: () => {
        currentSetOpenIndex(idx);
      },
      headingRef: headingRefs[idx],
      scrollableHeight,
    });
  });

  return <div className="accordion" ref={containerRef}>{menus}</div>;
}

export default Accordion;
