/**
 * Unit tests for LowerThirdOverlay component
 * Tests pure presentation component rendering
 */

import React from 'react';
import { render, screen } from '@testing-library/react';
import LowerThirdOverlay from '../Components/LowerThirdOverlay';

describe('LowerThirdOverlay', () => {
  test('renders header text when provided', () => {
    render(
      <LowerThirdOverlay
        header="Test Header"
        text="Test text"
        qr=""
        visible={true}
      />
    );

    expect(screen.getByRole('heading', { level: 1 })).toHaveTextContent('Test Header');
  });

  test('renders body text when provided', () => {
    render(
      <LowerThirdOverlay
        header="Header"
        text="Body text content"
        qr=""
        visible={true}
      />
    );

    const emElement = screen.getByText('Body text content');
    expect(emElement.tagName).toBe('EM');
  });

  test('renders QR code image when URL provided', () => {
    render(
      <LowerThirdOverlay
        header="Header"
        text="Text"
        qr="https://example.com/qr.png"
        visible={true}
      />
    );

    const qrImage = screen.getByRole('img');
    expect(qrImage).toHaveAttribute('src', 'https://example.com/qr.png');
  });

  test('applies visible class when visible=true', () => {
    const { container } = render(
      <LowerThirdOverlay
        header="Header"
        text="Text"
        qr=""
        visible={true}
      />
    );

    const overlay = container.querySelector('.tv-third-overlay');
    expect(overlay).toHaveClass('visible');
  });

  test('does not apply visible class when visible=false', () => {
    const { container } = render(
      <LowerThirdOverlay
        header="Header"
        text="Text"
        qr=""
        visible={false}
      />
    );

    const overlay = container.querySelector('.tv-third-overlay');
    expect(overlay).not.toHaveClass('visible');
  });

  test('handles empty strings gracefully', () => {
    const { container } = render(
      <LowerThirdOverlay
        header=""
        text=""
        qr=""
        visible={true}
      />
    );

    // Should render container without crashing
    expect(container.querySelector('.tv-third-overlay')).toBeInTheDocument();
  });

  test('renders all elements together', () => {
    render(
      <LowerThirdOverlay
        header="News Alert"
        text="Breaking news story"
        qr="https://example.com/qr.png"
        visible={true}
      />
    );

    expect(screen.getByRole('heading', { level: 1 })).toHaveTextContent('News Alert');
    expect(screen.getByText('Breaking news story')).toBeInTheDocument();
    expect(screen.getByRole('img')).toHaveAttribute('src', 'https://example.com/qr.png');
  });
});

