# TV App Unit Tests

## Overview
This directory contains unit tests for the TV application. Tests are organized by component/module and follow the testing best practices outlined below.

## Running Tests

```bash
# Run all tests
npm test

# Run tests in watch mode
npm test -- --watch

# Run tests with coverage
npm test -- --coverage

# Run specific test file
npm test helpers.test.js
```

## Test Organization

### ✅ Implemented Tests

1. **helpers.test.js** - Pure utility functions
   - `getInSeconds()` - Timestamp parsing (HH:MM:SS, MM:SS, with frames)
   - `formatDuration()` - Duration formatting
   - `parseDuration()` - ISO 8601 duration parsing
   - Cookie utilities

2. **tvStateMachine.test.js** - State management
   - Initial state
   - State transitions
   - Subscribe/unsubscribe
   - First/next/previous item calculations

3. **ErrorBoundary.test.js** - Error boundary component
   - Catches errors in child components
   - Displays fallback UI

### 📋 Planned Tests (TODO)

#### High Priority - Component Tests

4. **TODO: Player.test.js** - Core playback coordinator
   - [ ] Resume position calculation from localStorage
   - [ ] Progress tracking within clip boundaries
   - [ ] Auto-advance at endTs
   - [ ] Delay before progress updates (3s)
   - [ ] Pass startSeconds to YouTube component
   - [ ] Handle missing/invalid localStorage data

5. **TODO: YouTube.test.js** - YouTube wrapper
   - [ ] Initialize YouTube iframe API
   - [ ] Load video with startSeconds parameter
   - [ ] Mute state persistence (userHasUnmutedRef)
   - [ ] Emit state change events
   - [ ] Handle API loading errors
   - [ ] Forever-loop detection

6. **TODO: Controls.test.js** - Playback controls
   - [ ] Play/pause button interactions
   - [ ] Mute/unmute with localStorage persistence
   - [ ] Fullscreen toggle
   - [ ] Keyboard shortcut handlers (via Keys events)
   - [ ] Disabled states

7. **TODO: ChannelItems.test.js** - Playlist UI
   - [ ] Render playlist items
   - [ ] Display progress bars with correct percentages
   - [ ] Handle item click to change video
   - [ ] Progress bar click to seek
   - [ ] Reset progress button
   - [ ] Poll localStorage for progress updates

8. **TODO: Keys.test.js** - Keyboard navigation
   - [ ] Arrow key navigation (up/down/left/right)
   - [ ] Enter/space to select/play
   - [ ] Media keys (play/pause, prev/next)
   - [ ] Scrubbing (J/L, Shift+arrows)
   - [ ] Fullscreen/mute shortcuts (F/M)
   - [ ] Navigation map building
   - [ ] Reset idle state on keypress

9. **TODO: Mouse.test.js** - Activity tracking
   - [ ] Track mouse enter/leave
   - [ ] Idle timer (5 seconds)
   - [ ] Reset idle on mouse movement
   - [ ] Reset idle on keyboard activity

#### Medium Priority - Integration Tests

10. **TODO: TvContainer.integration.test.js** - Full app integration
    - [ ] Load channels and display channel list
    - [ ] Select channel and load playlist
    - [ ] Play video and track progress
    - [ ] Switch between videos in playlist
    - [ ] Auto-advance to next qualifying item
    - [ ] Resume playback from saved progress
    - [ ] Fullscreen mode toggle
    - [ ] Lower third overlay display

11. **TODO: clipBoundaries.integration.test.js** - Clip boundary handling
    - [ ] Progress calculated within clip boundaries
    - [ ] Seeking constrained to [startTs, endTs]
    - [ ] Auto-advance at endTs (not video end)
    - [ ] Resume position respects clip boundaries
    - [ ] Progress bar seeking within boundaries

#### Lower Priority - Hook Tests

12. **TODO: useChannelFetch.test.js** - Channel data fetching hook
13. **TODO: useChannelStateSync.test.js** - State machine sync hook
14. **TODO: useChannelApi.test.js** - API calls hook

## Testing Patterns

### Mocking localStorage

```javascript
beforeEach(() => {
  global.localStorage = {
    getItem: jest.fn(),
    setItem: jest.fn(),
    removeItem: jest.fn(),
    clear: jest.fn(),
  };
});
```

### Mocking tvStateMachine

```javascript
jest.mock('../tvStateMachine', () => ({
  getState: jest.fn(() => ({ currentItem: 0, playerState: 'paused' })),
  transition: jest.fn(),
  subscribe: jest.fn(() => jest.fn()), // returns unsubscribe
}));
```

### Mocking YouTube API

```javascript
global.YT = {
  Player: jest.fn().mockImplementation(() => ({
    loadVideoById: jest.fn(),
    playVideo: jest.fn(),
    pauseVideo: jest.fn(),
    seekTo: jest.fn(),
    getCurrentTime: jest.fn(() => 0),
    getDuration: jest.fn(() => 100),
    mute: jest.fn(),
    unMute: jest.fn(),
  })),
  PlayerState: {
    PLAYING: 1,
    PAUSED: 2,
    ENDED: 0,
  },
};
```

### Testing Async Effects

```javascript
import { waitFor } from '@testing-library/react';

test('updates after async effect', async () => {
  render(<Component />);
  
  await waitFor(() => {
    expect(screen.getByText('Loaded')).toBeInTheDocument();
  });
});
```

### Testing User Interactions

```javascript
import { fireEvent } from '@testing-library/react';

test('handles click', () => {
  const handleClick = jest.fn();
  render(<Button onClick={handleClick} />);
  
  fireEvent.click(screen.getByRole('button'));
  expect(handleClick).toHaveBeenCalledTimes(1);
});
```

## Test Coverage Goals

- **Utilities (helpers.js):** 100% coverage ✅
- **State Machine:** 90%+ coverage ✅
- **Components:** 70%+ coverage (in progress)
- **Integration:** Key user flows covered

## Notes

- Tests use Jest as the test runner (configured via react-scripts)
- React Testing Library for component tests
- Prefer testing user behavior over implementation details
- Mock external dependencies (YouTube API, localStorage, fetch)
- Keep tests focused and fast (unit tests should be < 100ms each)

## Contributing

When adding new features:
1. Write tests for pure functions first (easiest)
2. Write component tests with user interaction focus
3. Update this README with test descriptions
4. Aim for at least 70% coverage of new code
5. Mark complex tests as TODO if needed to unblock development
