<?php

namespace Drupal\tv\ChannelItemProvider;

use Drupal\tv\Entity\TvChannel;
use Symfony\Component\DependencyInjection\Attribute\AutowireIterator;

/**
 * Collects and manages TV channel item providers.
 *
 * This service collects all services tagged with 'tv.channel_item_provider'
 * and aggregates their media IDs for a given channel.
 */
class TvChannelItemProviderCollector {

  /**
   * @var TvChannelItemProviderInterface[]
   */
  protected array $providers = [];

  public function __construct(
    #[AutowireIterator('tv.channel_item_provider')]
    iterable $providers,
  ) {
    foreach ($providers as $provider) {
      $this->providers[] = $provider;
    }
    // Sort providers by weight.
    usort($this->providers, fn($a, $b) => $a->getWeight() <=> $b->getWeight());
  }

  /**
   * Collects all media IDs from all providers.
   *
   * @param \Drupal\tv\Entity\TvChannel $channel
   *   The TV channel.
   *
   * @return int[]
   *   Array of unique media entity IDs.
   */
  public function collectMediaIds(TvChannel $channel): array {
    $allIds = [];
    foreach ($this->providers as $provider) {
      $ids = $provider->getMediaIds($channel);
      $allIds = array_merge($allIds, $ids);
    }
    return array_unique($allIds);
  }

  /**
   * Returns all registered providers.
   *
   * @return TvChannelItemProviderInterface[]
   *   Array of providers sorted by weight.
   */
  public function getProviders(): array {
    return $this->providers;
  }

}
