<?php
/**
 * LowerThirdsList.php
 * Temporary array provider for lower thirds overlay data.
 * Replace with Drupal route/controller later.
 */

namespace Drupal\tv;

class LowerThirdsList {
  /**
   * Get lower thirds overlay data.
   *
   * @return array
   */
  /**
   * Collect lower third items from all tagged service providers (Drupal 11 style).
   *
   * @return array
   *   An array of lower third items from all providers.
   */
  /**
   * Get lower thirds overlay data for a specific media item.
   *
   * @param \Drupal\media\MediaInterface $item
   *   The media item entity.
   * @return array
   *   An array of lower third items.
   */
  public static function getItemList($item = NULL) {
    $headings = [];
    $region_url = NULL;
    // If this is a clip, try to get parent remote_video entity and its regions
    if ($item && $item->bundle() === 'clip' && $item->hasField('field_source_media') && !$item->get('field_source_media')->isEmpty()) {
      $parent = $item->get('field_source_media')->entity;
      if ($parent) {
        // Try to get region from parent
        if ($parent->hasField('field_regions') && !$parent->get('field_regions')->isEmpty()) {
          $region = $parent->get('field_regions')->entity;
          if ($region) {
            try {
              $urlObj = $region->toUrl('canonical', ['absolute' => TRUE]);
              $region_url = $urlObj->toString();
            } catch (\Exception $e) {}
          }
        }
        // Heading logic (unchanged)
        if ($parent->hasField('field_channel_title') && !$parent->get('field_channel_title')->isEmpty()) {
          $heading_value = $parent->get('field_channel_title')->value;
          if (mb_strlen($heading_value) > 0) {
            if (mb_strlen($heading_value) > 100) {
              $heading_value = mb_substr($heading_value, 0, 100) . '...';
            }
            $qr_path = self::generateQrForMedia($parent);
            $headings[] = [
              'heading' => $heading_value,
              'text' => 'Subscribe today!',
              'qr' => $qr_path,
            ];
          }
        }
      }
    } else if ($item && $item->hasField('field_channel_title') && !$item->get('field_channel_title')->isEmpty()) {
      $heading_value = $item->get('field_channel_title')->value;
      if (mb_strlen($heading_value) > 0) {
        if (mb_strlen($heading_value) > 100) {
          $heading_value = mb_substr($heading_value, 0, 100) . '...';
        }
        $qr_path = self::generateQrForMedia($item);
        $headings[] = [
          'heading' => $heading_value,
          'text' => 'subscribe today!',
          'qr' => $qr_path,
        ];
      }
      // Try to get region from item itself if not a clip
      if ($item->hasField('field_regions') && !$item->get('field_regions')->isEmpty()) {
        $region = $item->get('field_regions')->entity;
        if ($region) {
          try {
            $urlObj = $region->toUrl('canonical', ['absolute' => TRUE]);
            $region_url = $urlObj->toString();
          } catch (\Exception $e) {}
        }
      }
    }

    // Add region QR lower third if region_url found
    // @todo Extract this site-specific chunk & related logic. The domain-specific lower third should be provided by a module for that site.
    if (!empty($region_url)) {
      $headings[] = [
        'heading' => 'LittleTown.Pro TV',
        'text' => 'Find Your Pro',
        'qr' => self::generateQrForUrl($region_url),
      ];
    }

    return $headings;
  }

  /**
   * Generate a QR code image for a given URL.
   *
   * @param string $url
   *   The URL to encode in the QR code.
   * @return string
   *   The public path to the generated QR code image, or throbber if not available.
   */
  public static function generateQrForUrl($url) {
    $throbber_path = '/core/misc/throbber-active.gif';
    if (empty($url)) {
      return $throbber_path;
    }
    try {
      $qrDir = 'public://qr_codes';
      \Drupal::service('file_system')->prepareDirectory($qrDir, \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY);
      $filename = 'qr_' . md5($url) . '.png';
      $qrPath = $qrDir . '/' . $filename;
      $publicPath = '/sites/default/files/qr_codes/' . $filename;
      if (!file_exists(\Drupal::service('file_system')->realpath($qrPath))) {
        $result = (new \Endroid\QrCode\Builder\Builder(
          writer: new \Endroid\QrCode\Writer\PngWriter(),
          data: $url
        ))->build();
        $result->saveToFile(\Drupal::service('file_system')->realpath($qrPath));
      }
      return $publicPath;
    } catch (\Exception $e) {
      return $throbber_path;
    }
  }

  /**
   * Generate a QR code image for the remote video URL of a media item.
   *
   * @param \Drupal\media\MediaInterface|null $item
   *   The media item entity.
   * @return string
   *   The public path to the generated QR code image, or throbber if not available.
   */
  public static function generateQrForMedia($item = NULL) {
    $throbber_path = '/core/misc/throbber-active.gif';
    if (!$item || !$item->hasField('field_media_oembed_video') || $item->get('field_media_oembed_video')->isEmpty()) {
      return $throbber_path;
    }
    $url = $item->get('field_media_oembed_video')->value;
    if (empty($url)) {
      return $throbber_path;
    }
    return self::generateQrForUrl($url);
  }

  /**
   * Get lower thirds overlay data for a specific channel node.
   *
   * @param \Drupal\node\NodeInterface $channel
   *   The channel node entity.
   * @return array
   *   An array of lower third items.
   */
  public static function getChannelList($channel = NULL) {
    if ($channel && method_exists($channel, 'label')) {
      $text = $channel->label();
    } else {
      $text = 'Untitled Channel';
    }
    return [
      [
        'heading' => '',
        'text' => $text,
        'qr' => '',
      ],
    ];
  }
}
