# Twenty CRM Integration

A comprehensive Drupal module that provides seamless integration with the Twenty CRM API for customer relationship management, featuring advanced Tagify-based UI components for person and company reference fields.

## Table of Contents

- [Features](#features)
- [Requirements](#requirements)
- [Installation](#installation)
- [Configuration](#configuration)
- [Usage](#usage)
- [Field Types](#field-types)
- [API Services](#api-services)
- [Architecture](#architecture)
- [Working with Other Entity Types](#working-with-other-entity-types)
- [Error Handling](#error-handling)
- [Security](#security)
- [Troubleshooting](#troubleshooting)
- [Contributing](#contributing)
- [Sponsorship](#sponsorship)

## Features

- **Secure API Integration**: Secure API key management using Drupal's Key module
- **Reference Fields**: Advanced Tagify-based field widgets for person and company references
- **Real-time Search**: Auto-complete functionality with live Twenty CRM data
- **Multi-entity Support**: Person (contact) and company management
- **Configurable Endpoints**: Support for different Twenty CRM instances
- **Bearer Token Authentication**: Secure OAuth 2.0 compliant authentication
- **Comprehensive Logging**: Detailed error handling and logging
- **Flexible Data Access**: Built on standalone PHP package with DynamicEntity and helper classes
- **Drupal Integration**: Native Drupal services and configuration management

## Requirements

- **Drupal**: 9.5+ or 10+ or 11+
- **PHP**: 8.1 or higher
- **Required Modules**:
  - [Key](https://www.drupal.org/project/key) - Secure API key storage
  - [Tagify](https://www.drupal.org/project/tagify) - Advanced tagging interface
- **PHP Package**: factorial-io/twenty-crm-php-client (installed via Composer)

## Installation

### Via Composer (Recommended)

```bash
composer require drupal/twenty_crm
drush en twenty_crm
```

### Manual Installation

1. Download and place this module in `web/modules/contrib/twenty_crm/`
2. Install dependencies: `composer require factorial-io/twenty-crm-php-client`
3. Enable the module: `drush en twenty_crm`
4. Configure API credentials (see Configuration section)

## Configuration

### 1. API Key Setup

1. Navigate to **Configuration > System > Keys** (`/admin/config/system/keys`)
2. Click **Add key**
3. Set the following:
   - **Key ID**: `twenty_crm_api_key`
   - **Key name**: Twenty CRM API Key
   - **Key type**: Authentication
   - **Key provider**: Configuration
   - **Key value**: Your Twenty CRM API token

### 2. Module Configuration

Access the settings form via:
- **Configuration > Web Services > Twenty CRM** (`/admin/config/services/twenty-crm`)
- Module configuration link at `/admin/modules`

Configure the following:
- **Base URL**: Your Twenty CRM instance URL (e.g., `https://api.twenty.com/rest/`)
- **API Key**: Select the key created in step 1
- **Default Settings**: Timeout, cache settings, etc.

## Usage

### Reference Fields

The module provides custom field types for referencing Twenty CRM entities:

1. **Add Field**: Go to any content type and add a **Person Reference** or **Company Reference** field
2. **Configure Widget**: The fields use Tagify for auto-complete functionality
3. **Search & Select**: Users can search and select people or companies directly from Twenty CRM

### Programmatic Usage

#### Person Service (formerly Contact Service)

```php
use Drupal\twenty_crm\Helper\PersonHelper;
use Factorial\TwentyCrm\DTO\SearchOptions;
use Factorial\TwentyCrm\Query\FilterBuilder;

$personService = \Drupal::service('twenty_crm.person_service');

// Search people using FilterBuilder
$filter = FilterBuilder::create()
  ->where('emails.primaryEmail', 'eq', 'john@example.com')
  ->build();

$options = new SearchOptions(limit: 10);

$people = $personService->findPeople($filter, $options);

// Extract data using helper
foreach ($people as $person) {
  $name = PersonHelper::getFullName($person);
  $email = PersonHelper::getPrimaryEmail($person);
  $jobTitle = PersonHelper::getJobTitle($person);
}

// Get specific person by ID
$person = $personService->getPersonById('person-uuid');
```

#### Company Service

```php
use Drupal\twenty_crm\Helper\CompanyHelper;
use Factorial\TwentyCrm\DTO\SearchOptions;
use Factorial\TwentyCrm\Query\FilterBuilder;

$companyService = \Drupal::service('twenty_crm.company_service');

// Search companies using FilterBuilder
$filter = FilterBuilder::create()
  ->where('name', 'contains', 'Acme')
  ->build();

$options = new SearchOptions(limit: 10);

$companies = $companyService->findCompanies($filter, $options);

// Extract data using helper
foreach ($companies as $company) {
  $name = CompanyHelper::getName($company);
  $location = CompanyHelper::getLocationString($company);
}

// Get specific company by ID
$company = $companyService->getCompanyById('company-uuid');
```

#### Advanced Filtering

```php
use Factorial\TwentyCrm\Query\FilterBuilder;

// Multiple conditions (AND)
$filter = FilterBuilder::create()
  ->where('name.firstName', 'contains', 'John')
  ->where('company.id', 'eq', 'company-uuid-123')
  ->build();

// Search within a specific company
$filter = FilterBuilder::create()
  ->where('company.id', 'eq', $companyUuid)
  ->build();
```

#### Backwards Compatibility

For existing code, the old service names are still available as deprecated aliases:

```php
// Still works, but deprecated - use twenty_crm.person_service instead
$contactService = \Drupal::service('twenty_crm.contact_service');
```

## Field Types

### Person Reference Field

- **Field Type**: `customer_reference`
- **Widget**: Tagify-based auto-complete
- **Formatter**: Displays person name with optional company info
- **Storage**: Stores Twenty CRM person UUID, company UUID, and display names
- **Features**:
  - Real-time search across people
  - Multiple person selection
  - Cached results for performance
  - Validation against Twenty CRM data

### Company Reference Field

- **Field Type**: `company_reference`
- **Widget**: Tagify-based auto-complete
- **Formatter**: Displays company name with location info
- **Storage**: Stores Twenty CRM company UUID, name, city, and country
- **Features**:
  - Real-time search across companies
  - Multiple company selection
  - Cached results for performance
  - Validation against Twenty CRM data

## API Services

The module provides several Drupal services:

- `twenty_crm.client` - Core Twenty CRM client
- `twenty_crm.person_service` - Person (contact) management
- `twenty_crm.company_service` - Company management
- `twenty_crm.client_factory` - Client factory with configuration
- `twenty_crm.contact_service` - **(Deprecated)** Alias to person_service for backwards compatibility

## Architecture

### PHP Client Package

Built on [`factorial-io/twenty-crm-php-client`](https://github.com/factorial-io/twenty-crm-php-client) version 0.4+:
- Framework-agnostic API client
- PSR-18 HTTP client abstraction
- PSR-3 logging support
- Dynamic entity system with DynamicEntity
- Flexible FilterBuilder for queries
- Comprehensive error handling

### Drupal Integration Layer

- **Services**: Native Drupal service integration
- **Configuration**: Drupal configuration system
- **Security**: Key module integration
- **Caching**: Drupal cache integration
- **Logging**: Drupal logging system
- **Helper Classes**: `PersonHelper` and `CompanyHelper` for easy data extraction from DynamicEntity objects

### Helper Classes

The module provides helper classes to simplify working with DynamicEntity objects:

**PersonHelper**
- `getFullName()` - Get the person's full name
- `getFirstName()` - Get the first name
- `getLastName()` - Get the last name
- `getPrimaryEmail()` - Get the primary email address
- `getJobTitle()` - Get the job title
- `getCompanyId()` - Get the associated company ID

**CompanyHelper**
- `getName()` - Get the company name
- `getAddressCity()` - Get the city from address
- `getAddressCountry()` - Get the country from address
- `getLocationString()` - Get formatted location (e.g., "City, Country")

## Working with Other Entity Types

The underlying [`factorial-io/twenty-crm-php-client`](https://github.com/factorial-io/twenty-crm-php-client) package provides powerful capabilities for working with **any** Twenty CRM entity - not just Person and Company. This includes standard entities like Opportunity, Note, Task, and Activity, as well as any custom entities defined in your Twenty CRM instance.

### Dynamic Entity System

The PHP client uses a metadata-driven architecture that automatically adapts to your Twenty CRM schema. You can work with any entity without writing custom code:

```php
// Access any entity dynamically through the client
$client = \Drupal::service('twenty_crm.client');

// Work with opportunities
$opportunities = $client->entity('opportunity')->find($filter, $options);

// Work with notes
$notes = $client->entity('note')->find($filter, $options);

// Work with custom entities (e.g., "campaign")
$campaigns = $client->entity('campaign')->find($filter, $options);

foreach ($campaigns as $campaign) {
  echo $campaign->get('name') . "\n";
}
```

### Code Generation for Custom Services

For projects requiring type-safe access to additional entities, the PHP client includes a **code generator** that creates fully-typed entity classes, services, and collections from your Twenty CRM schema.

#### Step 1: Create Configuration File

Create `.twenty-codegen.yaml` in your project root:

```yaml
namespace: Drupal\my_module\TwentyCrm
output_dir: web/modules/custom/my_module/src/TwentyCrm
api_url: https://your-twenty.example.com/rest/
api_token: ${TWENTY_API_TOKEN}
entities:
  - opportunity
  - note
  - task
  - campaign  # Custom entities work too!
options:
  overwrite: true
```

#### Step 2: Generate Entity Classes

```bash
vendor/bin/twenty-generate --config=.twenty-codegen.yaml --with-services --with-collections
```

This generates typed PHP classes:

```
src/TwentyCrm/
├── Opportunity.php
├── OpportunityService.php
├── OpportunityCollection.php
├── Note.php
├── NoteService.php
├── NoteCollection.php
├── Task.php
├── TaskService.php
├── TaskCollection.php
├── Campaign.php
├── CampaignService.php
└── CampaignCollection.php
```

#### Step 3: Use Generated Classes

```php
use Drupal\my_module\TwentyCrm\Opportunity;
use Drupal\my_module\TwentyCrm\OpportunityService;
use Factorial\TwentyCrm\Query\FilterBuilder;
use Factorial\TwentyCrm\DTO\SearchOptions;

// Create service instance
$client = \Drupal::service('twenty_crm.client');
$opportunityService = new OpportunityService(
  $client->getHttpClient(),
  $client->registry()->getDefinition('opportunity')
);

// Search with full IDE autocomplete
$filter = FilterBuilder::create()
  ->equals('stage', 'QUALIFIED')
  ->greaterThan('amount', 10000)
  ->build();

$opportunities = $opportunityService->find($filter, new SearchOptions(limit: 20));

// Create new opportunity with type safety
$opportunity = new Opportunity($client->registry()->getDefinition('opportunity'));
$opportunity->setName('New Deal');
$opportunity->setAmount(50000);
$opportunity->setStage('PROSPECTING');

$created = $opportunityService->create($opportunity);
```

### Benefits of Code Generation

- **IDE Autocomplete**: Full support for property names and methods
- **Type Safety**: Compile-time error checking with PHPStan/Psalm
- **Custom Entity Support**: Works with any entity in your Twenty CRM instance
- **Entity Relations**: Automatic handling of relationships between entities

### When to Use Each Approach

| Approach | Use Case |
|----------|----------|
| **Dynamic Entities** | Quick prototyping, simple integrations, entities accessed infrequently |
| **Code Generation** | Production code, complex integrations, entities with many fields, team projects |

For more details, see the [PHP client documentation](https://github.com/factorial-io/twenty-crm-php-client).

## Error Handling

The module provides comprehensive error handling:

- **API Key Validation**: Ensures valid authentication
- **HTTP Error Handling**: Graceful handling of network issues
- **JSON Parsing**: Robust data parsing with fallbacks
- **User Feedback**: Clear error messages in the UI
- **Logging**: Detailed logs in the `twenty_crm` channel

### Debugging

Check logs at **Reports > Recent log messages** (`/admin/reports/dblog`) and filter by `twenty_crm`.

## Security

- **API Keys**: Securely stored using Drupal's Key module
- **HTTPS**: All requests use HTTPS by default
- **Authentication**: Bearer token authentication following OAuth 2.0 standards
- **Validation**: Input validation on all API calls
- **Permissions**: Drupal permission system integration

## Troubleshooting

### Common Issues

1. **API Key Not Found**
   - Ensure key is created with ID `twenty_crm_api_key`
   - Verify key has the correct API token

2. **Connection Issues**
   - Check base URL configuration
   - Verify network connectivity
   - Check firewall settings

3. **Tagify Not Loading**
   - Ensure Tagify module is enabled
   - Clear cache: `drush cr`
   - Check JavaScript console for errors

### Support

- **Issue Queue**: [Drupal.org issue queue](https://www.drupal.org/project/issues/twenty_crm)
- **Documentation**: [Project page](https://www.drupal.org/project/twenty_crm)

## Contributing

We welcome contributions! Please:

1. Check the [issue queue](https://www.drupal.org/project/issues/twenty_crm) for existing issues
2. Follow [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
3. Create merge requests following [GitLab contribution guidelines](https://www.drupal.org/drupalorg/docs/gitlab-integration/contributing)
4. Test your changes thoroughly

## Sponsorship

Development of this module is sponsored by **[Factorial.io](https://www.factorial.io)** - a leading Drupal development agency specializing in enterprise web solutions and API integrations.

Factorial.io provides expert Drupal consulting, custom development, and digital transformation services. Visit [factorial.io](https://www.factorial.io) to learn more about their services and how they can help with your next project.

---

**Maintainers**: Factorial.io Development Team  
**License**: GPL-2.0-or-later  
**Project Page**: https://www.drupal.org/project/twenty_crm