(function ($, Drupal, drupalSettings, once) {
  'use strict';

  Drupal.behaviors.twentyCrmTagifyElement = {
    attach: function (context, settings) {
      // Initialize Twenty CRM Tagify elements
      once('twenty-crm-tagify', '.twenty-crm-tagify', context).forEach(function (element) {
        initializeTwentyCrmTagify(element);
      });
    }
  };

  /**
   * Initialize Twenty CRM Tagify functionality.
   */
  function initializeTwentyCrmTagify(input) {
    const $input = $(input);
    const autocompleteUrl = $input.data('autocomplete-url');

    if (!autocompleteUrl) {
      console.error('No autocomplete URL found for Twenty CRM Tagify element');
      return;
    }

    // Check if this is a company or contact field
    const isCompanyField = $input.hasClass('customer-reference-company');
    const isContactField = $input.hasClass('customer-reference-contact');

    // Initialize Tagify with custom settings for Twenty CRM data
    const tagify = new Tagify(input, {
      maxTags: parseInt($input.data('max-items')) || 1,
      enforceWhitelist: false,
      tagTextProp: 'name', // What user sees in the tag
      valueKey: 'uuid',    // What gets stored as the field value
      dropdown: {
        enabled: parseInt($input.data('suggestions-dropdown')) || 1,
        maxItems: 10,
        closeOnSelect: true,
        highlightFirst: true
      },
      whitelist: [],
      templates: {
        tag: function(tagData) {
          return `
            <tag title="${tagData.name || tagData.value}"
                 contenteditable="false"
                 spellcheck="false"
                 tabIndex="-1"
                 class="tagify__tag ${tagData.class || ''}"
                 ${this.getAttributes(tagData)}>
              <x title="" class="tagify__tag__removeBtn" role="button" aria-label="remove tag"></x>
              <div>
                <span class="tagify__tag-text">${tagData.name || tagData.value}</span>
              </div>
            </tag>
          `;
        },
        dropdownItem: function(tagData) {
          const highlightedValue = tagData.searchByValue ? this.dropdown.highlight(tagData.value) : tagData.value;
          return `
            <div ${this.getAttributes(tagData)}
                 class="tagify__dropdown__item ${tagData.class || ''}"
                 tabindex="0"
                 role="option">
              <div class="tagify__dropdown__item__text">
                <strong>${highlightedValue}</strong>
                ${tagData.jobTitle ? `<div class="tagify__dropdown__item__sub">${tagData.jobTitle}</div>` : ''}
                ${tagData.location ? `<div class="tagify__dropdown__item__sub">${tagData.location}</div>` : ''}
                ${tagData.email ? `<div class="tagify__dropdown__item__sub">${tagData.email}</div>` : ''}
              </div>
            </div>
          `;
        }
      }
    });

    // Set up autocomplete
    let controller = new AbortController();

    tagify.on('input', function(e) {
      const value = e.detail.value.trim();
      if (value.length < 2) {
        tagify.loading(false);
        return;
      }

      // Use current autocomplete URL (may be updated by company selection)
      const currentUrl = $input.data('current-autocomplete-url') || autocompleteUrl;

      // Skip if URL contains __none__ (no company selected)
      if (currentUrl.includes('__none__')) {
        tagify.loading(false);
        return;
      }

      // Abort previous request
      controller.abort();
      controller = new AbortController();

      // Show loading animation
      tagify.loading(true);

      fetch(currentUrl + '?q=' + encodeURIComponent(value), {
        signal: controller.signal
      })
        .then(response => response.json())
        .then(data => {
          const suggestions = data.map(item => ({
            value: item.name,
            name: item.name,
            uuid: item.uuid,
            jobTitle: item.jobTitle || '',
            location: item.location || '',
            email: item.email || '',
            city: item.city || '',
            country: item.country || ''
          }));

          tagify.whitelist = suggestions;
          tagify.loading(false).dropdown.show(value);
        })
        .catch(error => {
          tagify.loading(false);
          if (error.name !== 'AbortError') {
            console.error('Twenty CRM autocomplete error:', error);
          }
        });
    });

    // Handle company/contact field dependency
    if (isCompanyField) {
      setupCompanyFieldBehavior(tagify, $input);
    } else if (isContactField) {
      setupContactFieldBehavior(tagify, $input);
    }

    // Store Tagify instance on the element
    input.tagify = tagify;
  }

  /**
   * Setup company field specific behavior.
   */
  function setupCompanyFieldBehavior(tagify, $input) {
    const wrapper = $input.closest('[id*="customer-reference-wrapper"]');
    const contactField = wrapper.find('.customer-reference-contact');

    tagify.on('add', function(e) {
      const tagData = e.detail.data;

      if (tagData.uuid) {
        // Clear contact selection
        const contactTagify = contactField[0] && contactField[0].tagify;
        if (contactTagify) {
          contactTagify.removeAllTags();
        }

        // Update contact field autocomplete URL
        const newContactUrl = '/twenty-crm/autocomplete/contact/' + tagData.uuid;
        contactField.attr('data-autocomplete-url', newContactUrl);

        // Enable contact field and update its internal autocomplete URL
        if (contactTagify) {
          contactTagify.setReadonly(false);
          // Store the new URL for our autocomplete logic
          contactField.data('current-autocomplete-url', newContactUrl);
        }
      }
    });

    tagify.on('remove', function(e) {
      // Clear and disable contact field
      const contactTagify = contactField[0] && contactField[0].tagify;
      if (contactTagify) {
        contactTagify.removeAllTags();
        contactTagify.setReadonly(true);
      }

      // Reset contact field URL to disabled state
      contactField.data('current-autocomplete-url', '/twenty-crm/autocomplete/contact/__none__');
    });
  }

  /**
   * Setup contact field specific behavior.
   */
  function setupContactFieldBehavior(tagify, $input) {
    // Initialize with current autocomplete URL
    const initialUrl = $input.data('autocomplete-url');
    $input.data('current-autocomplete-url', initialUrl);

    // Check if field should be disabled initially
    const isDisabled = $input.data('disabled') === 'true';
    if (isDisabled) {
      tagify.setReadonly(true);
    }

    tagify.on('add', function(e) {
      const tagData = e.detail.data;
    });

    tagify.on('remove', function(e) {
    });
  }

})(jQuery, Drupal, drupalSettings, once);
