<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Controller;

use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\twenty_crm\Helper\CompanyHelper;
use Drupal\twenty_crm\Helper\PersonHelper;
use Drupal\twenty_crm\TwentyCrmCompanyService;
use Drupal\twenty_crm\TwentyCrmPersonService;
use Factorial\TwentyCrm\DTO\SearchOptions;
use Factorial\TwentyCrm\Query\FilterBuilder;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Twenty CRM autocomplete functionality.
 */
class AutocompleteController extends ControllerBase {

  /**
   * Constructs an AutocompleteController object.
   */
  public function __construct(
    private readonly TwentyCrmCompanyService $companyService,
    private readonly TwentyCrmPersonService $personService,
  ) {}

  /**
   * Company autocomplete callback.
   */
  public function companyAutocomplete(Request $request): CacheableJsonResponse {
    $query = $request->query->get('q', '');

    if (\strlen($query) < 2) {
      return $this->createCachedResponse([]);
    }

    $filter = FilterBuilder::create()
      ->ilike('name', '%' . $query . '%')
      ->build();

    $options = new SearchOptions(limit: 10);

    $results = $this->companyService->findCompanies($filter, $options);

    if ($results->isEmpty()) {
      return $this->createCachedResponse([], ['twenty_crm:company_search']);
    }

    $suggestions = [];

    foreach ($results as $company) {
      $suggestions[] = [
        'uuid' => $company->getId(),
        'name' => CompanyHelper::getName($company),
        'location' => CompanyHelper::getLocationString($company),
        'city' => CompanyHelper::getAddressCity($company) ?? '',
        'country' => CompanyHelper::getAddressCountry($company) ?? '',
      ];
    }

    return $this->createCachedResponse($suggestions, ['twenty_crm:company_search']);
  }

  /**
   * Person autocomplete callback.
   */
  public function personAutocomplete(Request $request, string $company_uuid = ''): CacheableJsonResponse {
    $query = $request->query->get('q', '');

    if (\strlen($query) < 2 || $company_uuid === '__none__' || empty($company_uuid)) {
      return $this->createCachedResponse([]);
    }

    // Search by first name within the specified company.
    $filter = FilterBuilder::create()
      ->ilike('name.firstName', '%' . $query . '%')
      ->equals('companyId', $company_uuid)
      ->build();

    $options = new SearchOptions(limit: 10);

    $results = $this->personService->findPeople($filter, $options);

    if ($results->isEmpty()) {
      return $this->createCachedResponse([], ['twenty_crm:person_search']);
    }

    $suggestions = [];

    foreach ($results as $person) {
      $name = PersonHelper::getFullName($person);

      if (empty($name)) {
        $name = PersonHelper::getPrimaryEmail($person) ?? 'Unknown';
      }

      $suggestions[] = [
        'uuid' => $person->getId(),
        'name' => $name,
        'email' => PersonHelper::getPrimaryEmail($person) ?? '',
        'jobTitle' => PersonHelper::getJobTitle($person) ?? '',
      ];
    }

    return $this->createCachedResponse($suggestions, ['twenty_crm:person_search']);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('twenty_crm.company_service'),
      $container->get('twenty_crm.person_service'),
    );
  }

  /**
   * Create a cached JSON response.
   */
  private function createCachedResponse(array $data, array $cache_tags = []): CacheableJsonResponse {
    $response = new CacheableJsonResponse($data);
    $cache_metadata = new CacheableMetadata();
    $cache_metadata->setCacheMaxAge(15);
    $cache_metadata->setCacheContexts(['url.query_args:q', 'url.path']);
    if (!empty($cache_tags)) {
      $cache_metadata->setCacheTags($cache_tags);
    }
    $response->addCacheableDependency($cache_metadata);
    return $response;
  }

}
