<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\Textfield;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;

/**
 * Provides a Twenty CRM tagify autocomplete form element.
 *
 * Properties:
 * - #autocomplete_route: (required) The route name for autocomplete.
 * - #autocomplete_parameters: (optional) Parameters for the autocomplete route.
 * - #max_items: (optional) Maximum number of tags allowed. Defaults to 1.
 * - #placeholder: (optional) Placeholder text.
 * - #identifier: (optional) Unique identifier for this element.
 *
 * Usage example:
 * @code
 * $form['my_element'] = [
 *   '#type' => 'twenty_crm_tagify',
 *   '#title' => $this->t('Company'),
 *   '#autocomplete_route' => 'twenty_crm.company_autocomplete',
 *   '#max_items' => 1,
 *   '#placeholder' => 'Start typing company name...',
 *   '#identifier' => 'company-field',
 * ];
 * @endcode
 *
 * @FormElement("twenty_crm_tagify")
 */
class TwentyCrmTagify extends Textfield {

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $info = parent::getInfo();
    $class = static::class;

    $info['#maxlength'] = NULL;
    $info['#max_items'] = 1;
    $info['#autocomplete_route'] = NULL;
    $info['#autocomplete_parameters'] = [];
    $info['#identifier'] = '';
    $info['#placeholder'] = '';
    array_unshift($info['#process'], [$class, 'processTwentyCrmTagify']);

    return $info;
  }

  /**
   * Adds Twenty CRM tagify functionality to a form element.
   */
  public static function processTwentyCrmTagify(array &$element, FormStateInterface $form_state, array &$complete_form) {
    // Attach Tagify libraries and our custom enhancement.
    $element['#attached'] = [
      'library' => [
        'tagify/tagify',
        'tagify/default',
        'tagify/tagify_polyfils',
        'twenty_crm/twenty_crm_tagify_element',
      ],
    ];

    if (_tagify_is_gin_theme_active()) {
      $element['#attached']['library'][] = 'tagify/gin';
    }

    if (_tagify_is_claro_theme_active()) {
      $element['#attached']['library'][] = 'tagify/claro';
    }

    if (!isset($element['#attributes']['class'])) {
      $element['#attributes']['class'] = [];
    }
    // Don't use 'tagify-widget' to avoid conflicts with default Tagify behavior.
    $element['#attributes']['class'][] = 'twenty-crm-tagify';

    // Set Tagify-specific attributes.
    $element['#attributes']['data-max-items'] = $element['#max_items'];
    $element['#attributes']['data-suggestions-dropdown'] = 1;
    // CONTAINS.
    $element['#attributes']['data-match-operator'] = 1;
    $element['#attributes']['data-placeholder'] = $element['#placeholder'] ?? '';
    $element['#attributes']['data-identifier'] = $element['#identifier'] ?? '';
    $element['#attributes']['data-cardinality'] = $element['#max_items'];

    if (!isset($element['#attributes']['autocomplete'])) {
      $element['#attributes']['autocomplete'] = [];
    }
    $element['#attributes']['autocomplete'][] = 'off';

    // Build autocomplete URL.
    if (!empty($element['#autocomplete_route'])) {
      $parameters = $element['#autocomplete_parameters'] ?? [];
      $element['#attributes']['data-autocomplete-url'] = Url::fromRoute(
        $element['#autocomplete_route'],
        $parameters
      )->toString();
    }

    // Add Drupal settings for Tagify.
    $element['#attached']['drupalSettings']['tagify']['information_message'] = [
      'limit_tag' => (string) new TranslatableMarkup('Tags are limited to:'),
      'no_matching_suggestions' => (string) new TranslatableMarkup('No matching suggestions found for:'),
    ];

    return $element;
  }

}
