<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Twenty CRM settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The key repository service.
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->keyRepository = $container->get('key.repository');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'twenty_crm_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['twenty_crm.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('twenty_crm.settings');

    $form['api_configuration'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Configuration'),
      '#description' => $this->t('Configure the connection to your Twenty CRM instance.'),
    ];

    $form['api_configuration']['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#default_value' => $config->get('base_url') ?? 'https://api.twenty.com',
      '#description' => $this->t('The base URL for your Twenty CRM API instance (e.g., https://api.twenty.com or https://your-instance.twenty.com/api).'),
      '#required' => TRUE,
    ];

    $form['api_configuration']['api_key_id'] = [
      '#type' => 'select',
      '#title' => $this->t('API Key'),
      '#options' => $this->getKeyOptions(),
      '#default_value' => $config->get('api_key_id') ?? 'twenty_crm_api_key',
      '#description' => $this->t('Select the key that contains your Twenty CRM API token. You can manage keys at <a href=":url">Configuration > System > Keys</a>.', [
        ':url' => '/admin/config/system/keys',
      ]),
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $baseUrl = $form_state->getValue('base_url');

    // Validate that the base URL is a valid URL.
    if (!filter_var($baseUrl, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('base_url', $this->t('Please enter a valid URL.'));
    }

    // Validate that the selected key exists.
    $keyId = $form_state->getValue('api_key_id');
    $key = $this->keyRepository->getKey($keyId);
    if (!$key) {
      $form_state->setErrorByName('api_key_id', $this->t('The selected API key does not exist.'));
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('twenty_crm.settings');
    $config->set('base_url', rtrim($form_state->getValue('base_url'), '/'));
    $config->set('api_key_id', $form_state->getValue('api_key_id'));
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available key options for the select list.
   *
   * @return array
   *   An array of key options.
   */
  protected function getKeyOptions(): array {
    $options = [];
    $keys = $this->keyRepository->getKeys();

    foreach ($keys as $key) {
      $options[$key->id()] = $key->label();
    }

    if (empty($options)) {
      $options[''] = $this->t('- No keys available -');
    }

    return $options;
  }

}
