<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Helper;

use Factorial\TwentyCrm\Entity\DynamicEntity;

/**
 * Helper class for extracting data from Company DynamicEntity objects.
 */
final class CompanyHelper {

  /**
   * Get the name from a company entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $company
   *   The company entity.
   *
   * @return string
   *   The company name, or empty string if not available.
   */
  public static function getName(DynamicEntity $company): string {
    return $company->get('name') ?? '';
  }

  /**
   * Get the address city from a company entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $company
   *   The company entity.
   *
   * @return string|null
   *   The address city, or NULL if not available.
   */
  public static function getAddressCity(DynamicEntity $company): ?string {
    $address = $company->get('address');
    return $address?->getCity();
  }

  /**
   * Get the address country from a company entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $company
   *   The company entity.
   *
   * @return string|null
   *   The address country, or NULL if not available.
   */
  public static function getAddressCountry(DynamicEntity $company): ?string {
    $address = $company->get('address');
    return $address?->getCountry();
  }

  /**
   * Get a formatted location string from a company entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $company
   *   The company entity.
   *
   * @return string
   *   The location string (e.g., "City, Country"), or empty string.
   */
  public static function getLocationString(DynamicEntity $company): string {
    $parts = [];
    if ($city = self::getAddressCity($company)) {
      $parts[] = $city;
    }
    if ($country = self::getAddressCountry($company)) {
      $parts[] = $country;
    }
    return implode(', ', $parts);
  }

}
