<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Helper;

use Factorial\TwentyCrm\Entity\DynamicEntity;

/**
 * Helper class for extracting data from Person DynamicEntity objects.
 */
final class PersonHelper {

  /**
   * Get the full name from a person entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $person
   *   The person entity.
   *
   * @return string
   *   The full name, or empty string if not available.
   */
  public static function getFullName(DynamicEntity $person): string {
    $name = $person->get('name');
    return $name?->getFullName() ?? '';
  }

  /**
   * Get the first name from a person entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $person
   *   The person entity.
   *
   * @return string
   *   The first name, or empty string if not available.
   */
  public static function getFirstName(DynamicEntity $person): string {
    $name = $person->get('name');
    return $name?->getFirstName() ?? '';
  }

  /**
   * Get the last name from a person entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $person
   *   The person entity.
   *
   * @return string
   *   The last name, or empty string if not available.
   */
  public static function getLastName(DynamicEntity $person): string {
    $name = $person->get('name');
    return $name?->getLastName() ?? '';
  }

  /**
   * Get the primary email from a person entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $person
   *   The person entity.
   *
   * @return string|null
   *   The primary email, or NULL if not available.
   */
  public static function getPrimaryEmail(DynamicEntity $person): ?string {
    $emails = $person->get('emails');
    return $emails?->getPrimaryEmail();
  }

  /**
   * Get the job title from a person entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $person
   *   The person entity.
   *
   * @return string|null
   *   The job title, or NULL if not available.
   */
  public static function getJobTitle(DynamicEntity $person): ?string {
    return $person->get('jobTitle');
  }

  /**
   * Get the company ID from a person entity.
   *
   * @param \Factorial\TwentyCrm\Entity\DynamicEntity $person
   *   The person entity.
   *
   * @return string|null
   *   The company ID, or NULL if not available.
   */
  public static function getCompanyId(DynamicEntity $person): ?string {
    $company = $person->get('company');
    // The company field may be a string (UUID), an object with getId(), or a relation.
    if (is_string($company)) {
      return $company;
    }
    if (is_object($company) && method_exists($company, 'getId')) {
      return $company->getId();
    }
    // For relation objects that have an 'id' property.
    if (is_object($company) && property_exists($company, 'id')) {
      return $company->id;
    }
    return null;
  }

}
