<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\Field\FieldFormatter;

use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\twenty_crm\Helper\PersonHelper;
use Drupal\twenty_crm\TwentyCrmPersonService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'customer_reference_formatter' formatter.
 */
#[FieldFormatter(
  id: 'customer_reference_formatter',
  label: new TranslatableMarkup('Customer Reference Formatter'),
  field_types: ['customer_reference'],
)]
class CustomerReferenceFormatter extends FormatterBase {

  /**
   * Constructs a CustomerReferenceFormatter object.
   */
  public function __construct(
    string $plugin_id,
    mixed $plugin_definition,
    mixed $field_definition,
    array $settings,
    string $label,
    string $view_mode,
    array $third_party_settings,
    private readonly TwentyCrmPersonService $personService,
    private readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('twenty_crm.person_service'),
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'display_format' => 'full',
      'link_to_crm' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['display_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Display format'),
      '#default_value' => $this->getSetting('display_format'),
      '#options' => [
        'name_only' => $this->t('Contact name only'),
        'company_and_name' => $this->t('Company and contact name'),
        'full' => $this->t('Full information'),
      ],
    ];

    $elements['link_to_crm'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Link to Twenty CRM'),
      '#default_value' => $this->getSetting('link_to_crm'),
      '#description' => $this->t('Make contact names clickable links to the Twenty CRM system.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    $display_format_options = [
      'name_only' => $this->t('Contact name only'),
      'company_and_name' => $this->t('Company and contact name'),
      'full' => $this->t('Full information'),
    ];

    $summary[] = $this->t('Display: @format', ['@format' => $display_format_options[$this->getSetting('display_format')]]);

    if ($this->getSetting('link_to_crm')) {
      $summary[] = $this->t('Links to Twenty CRM');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      if (empty($item->contact_uuid)) {
        continue;
      }

      $display_format = $this->getSetting('display_format');
      $link_to_crm = $this->getSetting('link_to_crm');

      // Use stored values if available, otherwise try to fetch from CRM.
      $contact_name = $item->contact_name;
      $company_name = $item->company_name;

      // If we don't have cached display names, try to fetch from CRM.
      if (empty($contact_name)) {
        $person = $this->personService->getPersonById($item->contact_uuid);
        if ($person) {
          $contact_name = PersonHelper::getFullName($person);
          if (empty($contact_name)) {
            $contact_name = PersonHelper::getPrimaryEmail($person) ?? 'Unknown Contact';
          }
        }
      }

      if (empty($contact_name)) {
        $contact_name = 'Unknown Contact';
      }

      // Build display text based on format setting.
      $display_text = match ($display_format) {
        'name_only' => $contact_name,
        'company_and_name' => $company_name ? "{$company_name}: {$contact_name}" : $contact_name,
        'full' => $this->buildFullDisplay($item, $contact_name, $company_name),
        default => $contact_name,
      };

      if ($link_to_crm && $this->getTwentyCrmUrl()) {
        $elements[$delta] = [
          '#type' => 'link',
          '#title' => $display_text,
          '#url' => $this->buildTwentyCrmUrl($item->contact_uuid),
          '#attributes' => [
            'target' => '_blank',
            'rel' => 'noopener noreferrer',
          ],
        ];
      }
      else {
        $elements[$delta] = [
          '#markup' => $display_text,
        ];
      }
    }

    return $elements;
  }

  /**
   * Build full display format with all available information.
   */
  private function buildFullDisplay($item, string $contact_name, ?string $company_name): string {
    $parts = [];

    if ($company_name) {
      $parts[] = $company_name;
    }

    $parts[] = $contact_name;

    return implode(' - ', $parts);
  }

  /**
   * Get the Twenty CRM base URL from configuration.
   */
  private function getTwentyCrmUrl(): ?string {
    $config = $this->configFactory->get('twenty_crm.settings');
    return $config->get('base_url');
  }

  /**
   * Build Twenty CRM URL for a contact.
   */
  private function buildTwentyCrmUrl(string $contact_uuid): Url {
    $base_url = $this->getTwentyCrmUrl();
    if (!$base_url) {
      return Url::fromRoute('<nolink>');
    }

    // Convert REST API URL to frontend URL
    // From: https://factorial.twenty.com/rest
    // To: https://factorial.twenty.com/object/person/{uuid}
    $frontend_url = str_replace('/rest', '', $base_url);
    $contact_url = rtrim($frontend_url, '/') . '/object/person/' . $contact_uuid;

    return Url::fromUri($contact_url);
  }

}
