<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'customer_reference' field type.
 */
#[FieldType(
  id: 'customer_reference',
  label: new TranslatableMarkup('Twenty CRM Customer Reference'),
  default_formatter: 'customer_reference_formatter',
  default_widget: 'customer_reference_widget',
  description: new TranslatableMarkup('A reference to a Twenty CRM contact'),
)]
class CustomerReferenceItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName(): ?string {
    return 'contact_uuid';
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultStorageSettings(): array {
    return [
      'max_length' => 36,
      'is_ascii' => TRUE,
      'case_sensitive' => FALSE,
    ] + parent::defaultStorageSettings();
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties['contact_uuid'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Contact UUID'))
      ->setSetting('case_sensitive', $field_definition->getSetting('case_sensitive'))
      ->setRequired(FALSE);

    $properties['company_uuid'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Company UUID'))
      ->setSetting('case_sensitive', $field_definition->getSetting('case_sensitive'))
      ->setRequired(FALSE);

    $properties['company_name'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Company Name'))
      ->setRequired(FALSE);

    $properties['contact_name'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Contact Name'))
      ->setRequired(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = [
      'columns' => [
        'contact_uuid' => [
          'type' => 'varchar_ascii',
          'length' => 36,
          'not null' => TRUE,
        ],
        'company_uuid' => [
          'type' => 'varchar_ascii',
          'length' => 36,
          'not null' => FALSE,
        ],
        'company_name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
        'contact_name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'contact_uuid' => ['contact_uuid'],
        'company_uuid' => ['company_uuid'],
      ],
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition) {
    $random = new Random();
    $values['contact_uuid'] = $random->name(8, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(12, TRUE);
    $values['company_uuid'] = $random->name(8, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(4, TRUE) . '-' . $random->name(12, TRUE);
    $values['company_name'] = $random->sentences(2);
    $values['contact_name'] = $random->name();
    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty() {
    $value = $this->get('contact_uuid')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints() {
    $constraints = parent::getConstraints();

    // Only add the constraint if the field is required on the field instance level
    // This allows empty fields to pass validation during form building
    // but requires contact_uuid when saving non-empty field values.
    return $constraints;
  }

}
