<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'company_reference_widget' widget.
 */
#[FieldWidget(
  id: 'company_reference_widget',
  label: new TranslatableMarkup('Company Reference Widget'),
  field_types: ['company_reference'],
)]
class CompanyReferenceWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'placeholder' => 'Start typing company name...',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Placeholder'),
      '#default_value' => $this->getSetting('placeholder'),
      '#description' => $this->t('Text shown in the company autocomplete field.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if (!empty($this->getSetting('placeholder'))) {
      $summary[] = $this->t('Placeholder: @placeholder', ['@placeholder' => $this->getSetting('placeholder')]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];

    // Prepare default value as JSON object with UUID for existing data.
    $company_default = '';
    if (!empty($item->company_uuid) && !empty($item->company_name)) {
      $location_parts = [];
      if (!empty($item->company_city)) {
        $location_parts[] = $item->company_city;
      }
      if (!empty($item->company_country)) {
        $location_parts[] = $item->company_country;
      }
      $location = !empty($location_parts) ? ' (' . implode(', ', $location_parts) . ')' : '';

      $company_default = json_encode([
        [
          'uuid' => $item->company_uuid,
          'name' => $item->company_name,
          'value' => $item->company_name,
          'location' => $location,
          'city' => $item->company_city ?? '',
          'country' => $item->company_country ?? '',
        ],
      ]);
    }

    $element['company'] = [
      '#type' => 'twenty_crm_tagify',
      '#title' => $element['#title'] ?? $this->t('Company'),
      '#description' => $element['#description'] ?? '',
      '#default_value' => $company_default,
      '#attributes' => [
        'placeholder' => $this->getSetting('placeholder'),
        'class' => ['company-reference-company'],
        'data-autocomplete-url' => '/twenty-crm/autocomplete/company',
        'data-max-items' => 1,
      ],
      '#required' => $element['#required'] ?? FALSE,
    ];

    // Add value callback to handle form submissions.
    $element['#element_validate'][] = [$this, 'validateElement'];

    return $element;
  }

  /**
   * Validates the form element.
   */
  public function validateElement(&$element, FormStateInterface $form_state, &$complete_form) {
    $company_value = $element['company']['#value'] ?? '';

    if (!empty($company_value)) {
      $company_data = json_decode($company_value, TRUE);

      if (is_array($company_data) && !empty($company_data[0])) {
        $company = $company_data[0];

        // Store the values in the form state.
        $values = [
          'company_uuid' => $company['uuid'] ?? '',
          'company_name' => $company['name'] ?? '',
          'company_city' => $company['city'] ?? '',
          'company_country' => $company['country'] ?? '',
        ];

        $form_state->setValueForElement($element, $values);
      }
    }
    else {
      // Clear all values if no company is selected.
      $form_state->setValueForElement($element, [
        'company_uuid' => '',
        'company_name' => '',
        'company_city' => '',
        'company_country' => '',
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as &$value) {
      // Ensure all expected keys exist.
      $value += [
        'company_uuid' => '',
        'company_name' => '',
        'company_city' => '',
        'company_country' => '',
      ];
    }
    return $values;
  }

}
