<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'customer_reference_widget' widget.
 */
#[FieldWidget(
  id: 'customer_reference_widget',
  label: new TranslatableMarkup('Customer Reference Widget'),
  field_types: ['customer_reference'],
)]
class CustomerReferenceWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'company_placeholder' => 'Start typing company name...',
      'contact_placeholder' => 'Select company first, then contact...',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['company_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company placeholder'),
      '#default_value' => $this->getSetting('company_placeholder'),
      '#description' => $this->t('Text shown in the company autocomplete field.'),
    ];

    $elements['contact_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Contact placeholder'),
      '#default_value' => $this->getSetting('contact_placeholder'),
      '#description' => $this->t('Text shown in the contact autocomplete field.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];

    if (!empty($this->getSetting('company_placeholder'))) {
      $summary[] = $this->t('Company placeholder: @placeholder', ['@placeholder' => $this->getSetting('company_placeholder')]);
    }

    if (!empty($this->getSetting('contact_placeholder'))) {
      $summary[] = $this->t('Contact placeholder: @placeholder', ['@placeholder' => $this->getSetting('contact_placeholder')]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];

    $field_name = $this->fieldDefinition->getName();
    $wrapper_id = "customer-reference-wrapper-{$field_name}-{$delta}";

    $element['#type'] = 'container';
    $element['#attributes']['id'] = $wrapper_id;

    // Prepare default values as JSON objects with UUIDs for existing data.
    $company_default = '';
    if (!empty($item->company_uuid) && !empty($item->company_name)) {
      $company_default = json_encode([
        [
          'uuid' => $item->company_uuid,
          'name' => $item->company_name,
          'value' => $item->company_name,
        ],
      ]);
    }

    $contact_default = '';
    if (!empty($item->contact_uuid) && !empty($item->contact_name)) {
      $contact_default = json_encode([
        [
          'uuid' => $item->contact_uuid,
          'name' => $item->contact_name,
          'value' => $item->contact_name,
        ],
      ]);
    }

    // Company Tagify autocomplete field.
    $element['company'] = [
      '#type' => 'twenty_crm_tagify',
      '#title' => $this->t('Company'),
      '#default_value' => $company_default,
      '#placeholder' => $this->getSetting('company_placeholder'),
      '#autocomplete_route' => 'twenty_crm.company_autocomplete',
      '#max_items' => 1,
      '#identifier' => "company-tagify-{$field_name}-{$delta}",
      '#attributes' => [
        'data-company-uuid' => $item->company_uuid ?? '',
        'class' => ['customer-reference-company'],
      ],
    ];

    // Contact Tagify autocomplete field - depends on company selection.
    $company_uuid = $item->company_uuid ?? '';
    $contact_disabled = empty($company_uuid);

    $element['contact'] = [
      '#type' => 'twenty_crm_tagify',
      '#title' => $this->t('Contact'),
      '#default_value' => $contact_default,
      '#placeholder' => $contact_disabled
        ? $this->getSetting('contact_placeholder')
        : $this->t('Start typing contact name...'),
      '#autocomplete_route' => 'twenty_crm.contact_autocomplete',
      '#autocomplete_parameters' => [
        'company_uuid' => $company_uuid ?: '__none__',
      ],
      '#max_items' => 1,
      '#identifier' => "contact-tagify-{$field_name}-{$delta}",
      '#attributes' => [
        'data-contact-uuid' => $item->contact_uuid ?? '',
        'data-disabled' => $contact_disabled ? 'true' : 'false',
        'class' => ['customer-reference-contact'],
      ],
    ];

    // Note: JavaScript is attached automatically by the twenty_crm_tagify form element.
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as $delta => $value) {
      $contact_uuid = '';
      $company_uuid = '';
      $company_name = '';
      $contact_name = '';

      // Extract UUIDs directly from Tagify field values.
      if (!empty($value['company'])) {
        $company_data = json_decode($value['company'], TRUE);
        if (json_last_error() === JSON_ERROR_NONE && !empty($company_data[0])) {
          $company_uuid = $company_data[0]['uuid'] ?? '';
          $company_name = $company_data[0]['name'] ?? $company_data[0]['value'] ?? '';
        }
      }

      if (!empty($value['contact'])) {
        $contact_data = json_decode($value['contact'], TRUE);
        if (json_last_error() === JSON_ERROR_NONE && !empty($contact_data[0])) {
          $contact_uuid = $contact_data[0]['uuid'] ?? '';
          $contact_name = $contact_data[0]['name'] ?? $contact_data[0]['value'] ?? '';
        }
      }

      // Skip completely empty entries (both contact_uuid and company_uuid are empty).
      if (empty($contact_uuid) && empty($company_uuid)) {
        unset($values[$delta]);
        continue;
      }

      $values[$delta] = [
        'contact_uuid' => trim($contact_uuid),
        'company_uuid' => trim($company_uuid),
        'company_name' => trim($company_name),
        'contact_name' => trim($contact_name),
      ];
    }

    return $values;
  }

}
