<?php

declare(strict_types=1);

namespace Drupal\twenty_crm\Plugin\GraphQLCompose\FieldType;

use Drupal\Core\Field\FieldItemInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerItemInterface;
use Drupal\graphql_compose\Plugin\GraphQL\DataProducer\FieldProducerTrait;
use Drupal\graphql_compose\Plugin\GraphQLCompose\GraphQLComposeFieldTypeBase;
use Drupal\twenty_crm\Helper\CompanyHelper;
use Drupal\twenty_crm\Plugin\Field\FieldType\CompanyReferenceItem;

/**
 * {@inheritDoc}
 *
 * @GraphQLComposeFieldType(
 *   id = "company_reference",
 *   type_sdl = "TwentyCrmCompanyReference",
 * )
 */
class CompanyReferenceFieldType extends GraphQLComposeFieldTypeBase implements FieldProducerItemInterface {

  use FieldProducerTrait;

  /**
   * {@inheritdoc}
   */
  public function resolveFieldItem(FieldItemInterface $item, FieldContext $context) {
    if ($item instanceof CompanyReferenceItem) {
      $data = [
        'companyUuid' => $item->company_uuid,
        'companyName' => $item->company_name,
        'companyCity' => $item->company_city,
        'companyCountry' => $item->company_country,
      ];

      // If we have a UUID, fetch additional data from the CRM.
      if (!empty($item->company_uuid)) {
        $data['company'] = $this->getCompanyData($item->company_uuid);
      }

      return $data;
    }
  }

  /**
   * Get company data from Twenty CRM API.
   *
   * @param string $companyUuid
   *   The company UUID.
   *
   * @return array|null
   *   The company data or NULL if not found.
   */
  public static function getCompanyData(string $companyUuid): ?array {
    /** @var \Drupal\twenty_crm\TwentyCrmCompanyService $companyService */
    $companyService = \Drupal::service('twenty_crm.company_service');
    $company = $companyService->getCompanyById($companyUuid);

    if (!$company) {
      return NULL;
    }

    // Extract address data
    $address = $company->get('address');

    return [
      'id' => $company->getId(),
      'name' => CompanyHelper::getName($company),
      'addressCity' => $address?->getCity(),
      'addressCountry' => $address?->getCountry(),
      'addressStreet1' => $address?->getStreet1(),
      'addressStreet2' => $address?->getStreet2(),
      'addressState' => $address?->getState(),
      'addressPostcode' => $address?->getPostCode(),
      'employees' => $company->get('employees'),
      'locationString' => CompanyHelper::getLocationString($company),
      'createdAt' => $company->get('createdAt'),
      'updatedAt' => $company->get('updatedAt'),
    ];
  }

}
